﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.Linq;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using VeteransAffairs.Registries.BusinessAHOBPR;
using System.Reflection;
using System.Web.Script.Serialization;
using System.Globalization;
using System.Web;
using VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR
{
    public class AHOBPRJsonDataManager : AHOBPRBaseBO
    {
        const int UNKNOWN_SERVICE_STATUS_ID = 4;
        const string NOTFOUND = "NOT_FOUND";
        private AHOBPRShared _sharedManager = new AHOBPRShared();
        private AHOBPRRegistrantManager _registrantManager = new AHOBPRRegistrantManager();
        private AHOBPRFormResponseManager _formManager = new AHOBPRFormResponseManager();
        string _connectionInfo = string.Empty;
        public AHOBPRJsonDataManager() { }
        public AHOBPRJsonDataManager(string dbConnInfo)
        {
            _connectionInfo = dbConnInfo;
        }

        #region Get DoD Deployment Import data
        /// <summary>
        /// Get DoD Deployment Import data
        /// </summary>
        /// <param name="edipi"></param>
        /// <param name="ssn"></param>
        /// <returns></returns>
        public BprDoDDeploymentImports GetDoDDeploymentImports(string edipi, string ssn)
        {
            if (string.IsNullOrEmpty(edipi))
            {
                return null;
            }
            else
            {
                List<DOD_DEPLOYMENT_IMPORT> import = new List<DOD_DEPLOYMENT_IMPORT>();
                AHOBPRRegistrantManager registrantManager = new AHOBPRRegistrantManager();
                REGISTRANT registrant = registrantManager.GetRegistrantById(edipi, ssn);
                List<DOD_DEPLOYMENT_IMPORT> deployments = new List<DOD_DEPLOYMENT_IMPORT>();
                using (_dbAhobpr = GetLocalContext(_connectionInfo))
                {
                    try
                    {
                        SetLoadWithForDoDImports(_dbAhobpr);
                        deployments = (from e in _dbAhobpr.DOD_DEPLOYMENT_IMPORTs
                                       where (e.EDIPI == edipi)
                                       orderby e.DEPLOYMENT_START_DATE
                                       select e).ToList();
                    }
                    catch
                    {
                        deployments = new List<DOD_DEPLOYMENT_IMPORT>();
                    }
                }

                BprDoDDeploymentImports bprDoDDeploymentImports = new BprDoDDeploymentImports();
                if (deployments.Count > 0)
                {
                    bprDoDDeploymentImports.edipi = edipi;
                    bprDoDDeploymentImports.ssn = ssn;

                    if (registrant != null)
                    {
                        bprDoDDeploymentImports._id = registrant.REGISTRANT_ID.ToString();
                        bprDoDDeploymentImports.dob = registrant.BIRTH_DATE.HasValue ? Convert.ToDateTime(registrant.BIRTH_DATE).ToUniversalTime().ToString("yyyy'-'MM'-'dd'T'HH':'mm':'ss'.'fff'Z'") : string.Empty;
                        bprDoDDeploymentImports.firstName = registrant.FIRST_NAME;
                        bprDoDDeploymentImports.importId = registrant.REGISTRANT_ID.ToString();
                        bprDoDDeploymentImports.lastName = registrant.LAST_NAME;
                        bprDoDDeploymentImports.middleName = registrant.MIDDLE_NAME;

                    }
                    List<BprDoDDeployment> bprDeployments = new List<BprDoDDeployment>();
                    foreach (DOD_DEPLOYMENT_IMPORT deployment in deployments)
                    {
                        BprDoDDeployment bprDeployment = new BprDoDDeployment();
                        bprDeployment.branchOfService = deployment.SERVICE;
                        bprDeployment.deploymentCountryCode = deployment.LOCATION;
                        bprDeployment.deploymentStartDate = deployment.LOCATION_START_DATE.ToShortDateString();
                        bprDeployment.deploymentEndDate = deployment.LOCATION_END_DATE.ToShortDateString();
                        bprDeployment.dutyProjectCode = string.Empty;

                        bprDeployments.Add(bprDeployment);
                    }
                    bprDoDDeploymentImports.deployments = bprDeployments.ToArray();
                }
                else
                {
                    bprDoDDeploymentImports = null;
                }

                return bprDoDDeploymentImports;
            }
        }

        private void SetLoadWithForDoDImports(AHOBPRDataAccess db)
        {
            DataLoadOptions lo = new DataLoadOptions();
            lo.LoadWith<DOD_DEPLOYMENT_IMPORT>(e => e.LOCATION);
            lo.LoadWith<DOD_DEPLOYMENT_IMPORT>(e => e.PAY_PLAN_GRADE);

            db.LoadOptions = lo;
            db.DeferredLoadingEnabled = false;
        }

        #endregion

        #region Deployment location search
        /// <summary>
        /// Get deploymentLocations by search Criteria
        /// </summary>
        /// <param name="searchCriteria"></param>
        /// <returns></returns>
        public BprDeploymentLocation[] GetDeploymentLocations(string searchCriteria)
        {
            if (string.IsNullOrEmpty(searchCriteria))
            {
                return (new List<BprDeploymentLocation>()).ToArray();
            }
            else
            {
                List<STD_DEPLOYMENT_LOCATION> locations = new List<STD_DEPLOYMENT_LOCATION>();
                string[] searches = searchCriteria.Split(' ');
                using (_dbAhobpr = GetLocalContext(_connectionInfo))
                {
                    switch (searches.Length)
                    {
                        case 1:
                            locations = (from e in _dbAhobpr.STD_DEPLOYMENT_LOCATIONs
                                         where (e.NAMES.Contains(searches[0]) || e.COUNTRY.Contains(searches[0]) || e.BASE.Contains(searches[0])
                                                || e.PROVINCE.Contains(searches[0]) || e.OTHER_NAMES.Contains(searches[0]))
                                         orderby e.BASE, e.NAMES
                                         select e).ToList();
                            break;
                        case 2:
                            locations = (from e in _dbAhobpr.STD_DEPLOYMENT_LOCATIONs
                                         where (e.NAMES.Contains(searches[0]) || e.COUNTRY.Contains(searches[0]) || e.BASE.Contains(searches[0])
                                                || e.PROVINCE.Contains(searches[0]) || e.OTHER_NAMES.Contains(searches[0]))
                                             && (e.NAMES.Contains(searches[1]) || e.COUNTRY.Contains(searches[1]) || e.BASE.Contains(searches[1])
                                                || e.PROVINCE.Contains(searches[1]) || e.OTHER_NAMES.Contains(searches[1]))
                                         orderby e.BASE, e.NAMES
                                         select e).ToList();
                            break;
                        case 3:
                            locations = (from e in _dbAhobpr.STD_DEPLOYMENT_LOCATIONs
                                         where (e.NAMES.Contains(searches[0]) || e.COUNTRY.Contains(searches[0]) || e.BASE.Contains(searches[0])
                                                || e.PROVINCE.Contains(searches[0]) || e.OTHER_NAMES.Contains(searches[0]))
                                             && (e.NAMES.Contains(searches[1]) || e.COUNTRY.Contains(searches[1]) || e.BASE.Contains(searches[1])
                                                || e.PROVINCE.Contains(searches[1]) || e.OTHER_NAMES.Contains(searches[1]))
                                             && (e.NAMES.Contains(searches[2]) || e.COUNTRY.Contains(searches[2]) || e.BASE.Contains(searches[2])
                                                || e.PROVINCE.Contains(searches[2]) || e.OTHER_NAMES.Contains(searches[2]))
                                         orderby e.BASE, e.NAMES
                                         select e).ToList();
                            break;
                        case 4:
                            locations = (from e in _dbAhobpr.STD_DEPLOYMENT_LOCATIONs
                                         where (e.NAMES.Contains(searches[0]) || e.COUNTRY.Contains(searches[0]) || e.BASE.Contains(searches[0])
                                                || e.PROVINCE.Contains(searches[0]) || e.OTHER_NAMES.Contains(searches[0]))
                                             && (e.NAMES.Contains(searches[1]) || e.COUNTRY.Contains(searches[1]) || e.BASE.Contains(searches[1])
                                                || e.PROVINCE.Contains(searches[1]) || e.OTHER_NAMES.Contains(searches[1]))
                                            && (e.NAMES.Contains(searches[2]) || e.COUNTRY.Contains(searches[2]) || e.BASE.Contains(searches[2])
                                                || e.PROVINCE.Contains(searches[2]) || e.OTHER_NAMES.Contains(searches[2]))
                                            && (e.NAMES.Contains(searches[3]) || e.COUNTRY.Contains(searches[3]) || e.BASE.Contains(searches[3])
                                                || e.PROVINCE.Contains(searches[3]) || e.OTHER_NAMES.Contains(searches[3]))
                                         orderby e.BASE, e.NAMES
                                         select e).ToList();
                            break;
                        case 5:
                            locations = (from e in _dbAhobpr.STD_DEPLOYMENT_LOCATIONs
                                         where (e.NAMES.Contains(searches[0]) || e.COUNTRY.Contains(searches[0]) || e.BASE.Contains(searches[0])
                                                || e.PROVINCE.Contains(searches[0]) || e.OTHER_NAMES.Contains(searches[0]))
                                             && (e.NAMES.Contains(searches[1]) || e.COUNTRY.Contains(searches[1]) || e.BASE.Contains(searches[1])
                                                || e.PROVINCE.Contains(searches[1]) || e.OTHER_NAMES.Contains(searches[1]))
                                             && (e.NAMES.Contains(searches[2]) || e.COUNTRY.Contains(searches[2]) || e.BASE.Contains(searches[2])
                                                || e.PROVINCE.Contains(searches[2]) || e.OTHER_NAMES.Contains(searches[2]))
                                             && (e.NAMES.Contains(searches[3]) || e.COUNTRY.Contains(searches[3]) || e.BASE.Contains(searches[3])
                                                || e.PROVINCE.Contains(searches[3]) || e.OTHER_NAMES.Contains(searches[3]))
                                             && (e.NAMES.Contains(searches[4]) || e.COUNTRY.Contains(searches[4]) || e.BASE.Contains(searches[4])
                                                || e.PROVINCE.Contains(searches[4]) || e.OTHER_NAMES.Contains(searches[4]))
                                         orderby e.BASE, e.NAMES
                                         select e).ToList();
                            break;
                    }
                }

                List<BprDeploymentLocation> bprLocations = new List<BprDeploymentLocation>();
                foreach (STD_DEPLOYMENT_LOCATION location in locations)
                {
                    BprDeploymentLocation bprLocation = new BprDeploymentLocation();
                    bprLocation.@base = location.BASE;
                    bprLocation.names = location.NAMES;
                    bprLocation.otherNames = location.OTHER_NAMES;
                    bprLocation.province = location.PROVINCE;
                    bprLocation.country = location.COUNTRY;

                    bprLocations.Add(bprLocation);
                }

                return bprLocations.ToArray();
            }
        }
        #endregion

        #region Json Parser
        /// <summary>
        /// Parse JSON data and save it to SQL tables
        /// </summary>
        /// <param name="jsonQueueId"></param>
        /// <returns></returns>
        public bool ParseJsonData(int jsonQueueId)
        {
            bool result = false;
            JSON_QUEUE queue = GetJsonQueue(jsonQueueId);
            if (queue == null)
            {
                result = true;
            }
            else
            {
                int jsonId = queue.REGISTRANT_JSON_ID;
                REGISTRANT_JSON registrantJson = GetJsonDataById(jsonId);
                if (registrantJson == null)
                {
                    result = true;
                }
                else
                {
                    if (registrantJson.OBJECT_TYPE == AHOBPRGlobal.AhobprJsonDataTypeUser)
                    {
                        result = ParseUserJson(registrantJson.JSON_DATA, queue);
                    }
                    else if (registrantJson.OBJECT_TYPE == AHOBPRGlobal.AhobprJsonDataTypeInfo)
                    {
                        result = ParseInfoJson(registrantJson.JSON_DATA, queue);
                    }
                    else if (registrantJson.OBJECT_TYPE == AHOBPRGlobal.AhobprJsonDataTypeResponses)
                    {
                        result = ParseResponseJson(registrantJson.JSON_DATA, queue);
                    }
                    else if (registrantJson.OBJECT_TYPE == AHOBPRGlobal.AhobprJsonDataTypeForm)
                    {
                        result = ParseFormJson(registrantJson.JSON_DATA, queue);
                    }
                    else
                    {
                        result = true;
                    }

                }
            }

            return result;
        }


        public bool ParseJsonFollowupData(int jsonQueueFollowupId)
        {
            bool result = false;
            JSON_QUEUE_FOLLOWUP queueFollowup = GetJsonQueueFollowup(jsonQueueFollowupId);
            if (queueFollowup == null)
            {
                result = true;
            }
            else
            {
                int jsonFollowupId = queueFollowup.REGISTRANT_JSON_FOLLOWUP_ID;
                REGISTRANT_JSON_FOLLOWUP registrantJsonFollowup = GetJsonFollowupDataById(jsonFollowupId);
                if (registrantJsonFollowup == null)
                {
                    result = true;
                }
                else
                {
                    if (registrantJsonFollowup.OBJECT_TYPE == AHOBPRGlobal.AhobprJsonTypeResponsesFollowup)
                    {
                        result = ParseResponseJsonFollowup(registrantJsonFollowup.JSON_DATA, queueFollowup);
                    }
                    else
                    {
                        result = true;
                    }
                }
            }

            return result;
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="jsonData"></param>
        /// <returns></returns>
        public bool ParseUserJson(string jsonData)
        {
            bool result = false;
            try
            {
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                BprUser bprUser = serializer.Deserialize<BprUser>(jsonData);
                if (bprUser != null)
                {
                    REGISTRANT registrant = _registrantManager.GetRegistrantByUserId(bprUser.userId);
                    registrant = _registrantManager.InitializeRegistrantDBObject(registrant);


                    registrant.Snum = bprUser.ssn;
                    registrant.USER_ID = bprUser.userId;
                    SetRegistrantNameFields(bprUser, registrant, NOTFOUND);
                    registrant.EDIPI = bprUser.edipi == NOTFOUND ? string.Empty : bprUser.edipi;
                    if (!string.IsNullOrEmpty(bprUser.dob) && bprUser.dob != NOTFOUND)
                    {
                        // the birth date is UTC time
                        registrant.BIRTH_DATE = DateTimeOffset.Parse(bprUser.dob).UtcDateTime;
                    }
                    if (!string.IsNullOrEmpty(bprUser.userState.lastUpdated))
                    {
                        registrant.REGISTRANT_STATUS_LAST_UPDATED_DATE = DateTimeOffset.Parse(bprUser.userState.lastUpdated).UtcDateTime;
                    }
                    registrant.DECEASED_FLAG = false;
                    SetStdRegistrantStatus(bprUser, registrant);

                    if (!String.IsNullOrEmpty(bprUser.icn) && bprUser.icn != NOTFOUND)
                    {
                        registrant.ICN = bprUser.icn;
                    }
                    if (!String.IsNullOrEmpty(bprUser.gender) && bprUser.gender != NOTFOUND)
                    {
                        registrant.GENDER = bprUser.gender;
                    }
                    SetServiceStatus(registrant);
                    result = (_registrantManager.UpdateRegistrant(registrant) >= 0);
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }

            return result;
        }

        private static void SetRegistrantNameFields(BprUser bprUser, REGISTRANT registrant, string notFound)
        {
            registrant.FIRST_NAME = bprUser.firstName == notFound ? string.Empty : bprUser.firstName;
            registrant.LAST_NAME = bprUser.lastName == notFound ? string.Empty : bprUser.lastName;
            registrant.MIDDLE_NAME = bprUser.middleName == notFound ? string.Empty : bprUser.middleName;
            registrant.FIRST_NAME = registrant.FIRST_NAME.ToUpper();
            registrant.LAST_NAME = registrant.LAST_NAME.ToUpper();
            registrant.MIDDLE_NAME = registrant.MIDDLE_NAME.ToUpper();
        }

        private void SetStdRegistrantStatus(BprUser bprUser, REGISTRANT registrant)
        {
            int statusId = GetRegistrantStatusId(bprUser.userState.state);
            if (statusId > 0)
            {
                registrant.STD_REGISTRANT_STATUS_ID = statusId;
            }
        }

        private static void SetServiceStatus(REGISTRANT registrant)
        {
            var rssmanager = new RegistrantServiceStatusMapper();
            registrant.STD_SERVICE_STATUS_ID = rssmanager.GetServiceStatus(registrant.EDIPI);
            if (ServiceMode.Mode == "migration")
            {
                registrant.STD_SERVICE_STATUS_ID_WHEN_ENROLLED = new int?(UNKNOWN_SERVICE_STATUS_ID);
            }
            else
            {
                if (!registrant.STD_SERVICE_STATUS_ID_WHEN_ENROLLED.HasValue)
                {
                    registrant.STD_SERVICE_STATUS_ID_WHEN_ENROLLED = registrant.STD_SERVICE_STATUS_ID;
                }
            }
        }






        /// <summary>
        /// 
        /// </summary>
        /// <param name="jsonData"></param>
        /// <returns></returns>
        private bool ParseUserJson(string jsonData, JSON_QUEUE queue)
        {
            bool result = false;
            queue.SetAsChangeTrackingRoot();
            queue.SetAsUpdateOnSubmit();
            queue.PROCESS_START_DATE = DateTime.Now;
            try
            {
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                BprUser bprUser = serializer.Deserialize<BprUser>(jsonData);
                if (bprUser != null)
                {
                    result = ParseUserJson(jsonData);

                    queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                    if (result)
                    {
                        queue.PROCESS_STATUS = "success";
                        SaveJsonQueueData(queue);
                    }
                    else
                    {
                        queue.PROCESS_STATUS = "failed";
                        SaveJsonQueueData(queue);
                        SaveJsonData(bprUser.userId, AHOBPRGlobal.AhobprJsonDataTypeUser, jsonData, "Failed Save");
                    }
                }

            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "JSON QueueId: " + queue.JSON_QUEUE_ID.ToString() + ", Error: " + ex.Message);
                queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                queue.PROCESS_STATUS = "failed";
                queue.PROCESS_ERROR = ex.Message;
                SaveJsonQueueData(queue);
                result = false;
            }

            return result;
        }

        /// <summary>
        /// Get registrant status ID by the name in JSON
        /// </summary>
        /// <param name="status"></param>
        /// <returns></returns>
        private int GetRegistrantStatusId(string status)
        {
            int statusId = 0;

            switch (status)
            {
                case "ELIGIBLE":
                    statusId = (int)Enums.RegistrantStatus.Eligible;
                    break;
                case "NOT_ELIGIBLE":
                    statusId = (int)Enums.RegistrantStatus.NotEligible;
                    break;
                case "CONSENT":
                    statusId = (int)Enums.RegistrantStatus.Consent;
                    break;
                case "NOT_CONSENT":
                    statusId = (int)Enums.RegistrantStatus.NoConsent;
                    break;
                case "REVIEW_REQUESTED":
                    statusId = (int)Enums.RegistrantStatus.Review;
                    break;
                case "REVIEWED_NOT_ELIGIBLE":
                    statusId = (int)Enums.RegistrantStatus.ReviewedNotEligible;
                    break;
                case "PARTICIPANT":
                    statusId = (int)Enums.RegistrantStatus.Participant;
                    break;
                case "REVIEW":
                    statusId = (int)Enums.RegistrantStatus.Review;
                    break;
                case "REVIEWED_ELIGIBLE":
                    statusId = (int)Enums.RegistrantStatus.ReviewedEigible;
                    break;
            }

            return statusId;
        }



        /// <summary>
        /// 
        /// </summary>
        /// <param name="jsonData"></param>
        /// <returns></returns>
        private bool ParseInfoJson(string jsonData, JSON_QUEUE queue)
        {
            bool result = false;
            queue.SetAsChangeTrackingRoot();
            queue.SetAsUpdateOnSubmit();
            queue.PROCESS_START_DATE = DateTime.Now;
            try
            {
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                BprInfo bprInfo = serializer.Deserialize<BprInfo>(jsonData);
                if (bprInfo != null)
                {
                    REGISTRANT registrant = _registrantManager.GetRegistrantByUserIdWithInfo(bprInfo.userId);
                    if (registrant == null)
                    {
                        queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                        queue.PROCESS_STATUS = "failed";
                        queue.PROCESS_ERROR = "Registrant not found";
                        SaveJsonQueueData(queue);
                    }
                    else
                    {
                        MergeRegistrantPortalJsonWithExistingDatabaseInfo(bprInfo, registrant);
                        // Save Registrant Info
                        result = SaveRegistrantInfo(jsonData, queue, registrant);
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Parse JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "JSON QueueId: " + queue.JSON_QUEUE_ID.ToString() + ", Error: " + ex.Message);
                queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                queue.PROCESS_STATUS = "failed";
                queue.PROCESS_ERROR = ex.Message;
                SaveJsonQueueData(queue);
                result = false;
            }
            return result;
        }

        public void MergeRegistrantPortalJsonWithExistingDatabaseInfo(BprInfo bprInfo, REGISTRANT registrant)
        {
            registrant.SetAsChangeTrackingRoot();
            AddSSNToRegistrant(bprInfo, registrant);
            // Registrant Record Info
            AddRegistrantRecordInfotoRegistrant(bprInfo, registrant);
            //Registrant Contact Info
            AddContactInfoToRegistrant(bprInfo, registrant);
            AddAddressToRegistrant(bprInfo, registrant);
            AddEmailAddressToRegistrant(bprInfo, registrant);
            AddPhoneNumberToRegistrant(bprInfo, registrant);
            ProcessServiceEpisodes(bprInfo, registrant);
            //Deployments and Reconsidered Deployments
            ProcessDeployments(bprInfo, registrant);
            //Contact Preference:
            if (!string.IsNullOrEmpty(bprInfo.contactInfo.contactPreference))
            {
                registrant.CONTACT_PREFERENCE = bprInfo.contactInfo.contactPreference;
            }
            //Set the Registrant.STD_BRANCH_OF_SERVICE_ID, CURRENT & WHEN ENROLLED:
            //First, get the branch of service for the most recent service episode.
            //Set that as the current.
            int? mostRecentServiceEpisodeBranchOfServiceId = ProcessMostRecentBranchOfService(registrant);
            //Then, set the when enrolled branch of service:
            //If this is the data migration, we don't know what the enrolled branch of service is
            //so set it to -1:
            SetComponentAndBranchOFService(registrant, mostRecentServiceEpisodeBranchOfServiceId);
        }

        private void ProcessDeployments(BprInfo bprInfo, REGISTRANT registrant)
        {
            //Addds new deployments or updates existing DB entries with new data
            MergeJsonDeploymentDataWithAnyExistingRegistrantDeploymentDatabaseEntries(bprInfo, registrant);

            ProcessReconsideredDeployments(bprInfo, registrant);
            ProcessReconsideredDeploymentAddressInfo(bprInfo, registrant);
            ProcessReconsideredDeploymentEmailInfo(bprInfo, registrant);
            ProcessReconsideredDeploymentPhoneInfo(bprInfo, registrant);
        }

        private bool SaveRegistrantInfo(string jsonData, JSON_QUEUE queue, REGISTRANT registrant)
        {
            bool result = (_registrantManager.UpdateRegistrant(registrant) >= 0);
            queue.PROCESS_COMPLETE_DATE = DateTime.Now;
            if (result)
            {
                queue.PROCESS_STATUS = "success";
                SaveJsonQueueData(queue);
            }
            else
            {
                queue.PROCESS_STATUS = "failed";
                queue.PROCESS_ERROR = "Failed to save Registrant.";
                SaveJsonQueueData(queue);
                SaveJsonData(registrant.USER_ID, AHOBPRGlobal.AhobprJsonDataTypeInfo, jsonData, "Failed Save");
            }

            return result;
        }
        public static int? ProcessMostRecentBranchOfService(REGISTRANT registrant)
        {
            List<REGISTRANT_SERVICE_EPISODE> list =
                (from episodes in registrant.REGISTRANT_SERVICE_EPISODEs
                 where episodes.LINQEntityState != EntityState.Deleted
                 select episodes).ToList();
            //list.OrderByDescending(e => e.SERVICE_EPISODE_START_DATE);

            int? mostRecentServiceEpisodeBranchOfServiceId = null;
            if (list.Count > 0)
            {
                mostRecentServiceEpisodeBranchOfServiceId = list.OrderByDescending(e => e.SERVICE_EPISODE_START_DATE).First().STD_BRANCH_OF_SERVICE_ID;
            }

            registrant.STD_BRANCH_OF_SERVICE_ID_CURRENT = mostRecentServiceEpisodeBranchOfServiceId;
            return mostRecentServiceEpisodeBranchOfServiceId;
        }
        private static void SetComponentAndBranchOFService(REGISTRANT registrant, int? mostRecentServiceEpisodeBranchOfServiceId)
        {
            if (ServiceMode.Mode == "migration")
            {
                registrant.STD_BRANCH_OF_SERVICE_ID_WHEN_ENROLLED = (int)Enums.BranchOfService.Uknown;

                //Do the same for the component:
                registrant.STD_COMPONENT_ID_WHEN_ENROLLED = (int)Enums.Compo.NoDataAvailable;
            }
            else
            {
                //Otherwise, if this is the first time the JSON parser has been run for this registrant,
                //Set the when enrolled branch of service to the most recent service episode 
                //branch of service:
                if (registrant.STD_BRANCH_OF_SERVICE_ID_WHEN_ENROLLED == null)
                {
                    registrant.STD_BRANCH_OF_SERVICE_ID_WHEN_ENROLLED = mostRecentServiceEpisodeBranchOfServiceId;
                }
                //Otherwise, leave it alone.

                //Do the same for the component:
                if (registrant.STD_COMPONENT_ID_WHEN_ENROLLED == null)
                {
                    registrant.STD_COMPONENT_ID_WHEN_ENROLLED = registrant.STD_COMPONENT_ID_CURRENT;
                }
            }
        }

        private static void AddContactInfoToRegistrant(BprInfo bprInfo, REGISTRANT registrant)
        {
            if (registrant.REGISTRANT_CONTACT_INFO == null)
            {
                registrant.REGISTRANT_CONTACT_INFO = new REGISTRANT_CONTACT_INFO();
                registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ID = registrant.REGISTRANT_ID;
                registrant.REGISTRANT_CONTACT_INFO.SetAsInsertOnSubmit();
            }
            else
            {
                registrant.REGISTRANT_CONTACT_INFO.UPDATED = DateTime.Now;
                registrant.REGISTRANT_CONTACT_INFO.UPDATEDBY = "REST Service";
                registrant.REGISTRANT_CONTACT_INFO.SetAsUpdateOnSubmit();
            }
            registrant.REGISTRANT_CONTACT_INFO.FIRST_NAME_CONFIRMED = bprInfo.contactInfo.firstName.ToUpper();
            registrant.REGISTRANT_CONTACT_INFO.LAST_NAME_CONFIRMED = bprInfo.contactInfo.lastName.ToUpper();
            registrant.REGISTRANT_CONTACT_INFO.MIDDLE_NAME_CONFIRMED = bprInfo.contactInfo.middleName.ToUpper();
            registrant.REGISTRANT_CONTACT_INFO.PREVIOUS_VALUES = bprInfo.contactInfo.previousValues;
            if (!string.IsNullOrEmpty(bprInfo.contactInfo.previousUpdated))
            {
                registrant.REGISTRANT_CONTACT_INFO.PREVIOUS_UPDATED_DATE = DateTimeOffset.Parse(bprInfo.contactInfo.previousUpdated).UtcDateTime;
            }
        }

        private static void AddRegistrantRecordInfotoRegistrant(BprInfo bprInfo, REGISTRANT registrant)
        {
            if (registrant.REGISTRANT_RECORD_INFO == null)
            {
                registrant.REGISTRANT_RECORD_INFO = new REGISTRANT_RECORD_INFO();
                registrant.REGISTRANT_RECORD_INFO.REGISTRANT_ID = registrant.REGISTRANT_ID;
                registrant.REGISTRANT_RECORD_INFO.SetAsInsertOnSubmit();
            }
            else
            {
                registrant.REGISTRANT_RECORD_INFO.UPDATEDBY = "REST Service";
                registrant.REGISTRANT_RECORD_INFO.UPDATED = DateTime.Now;
                registrant.REGISTRANT_RECORD_INFO.SetAsUpdateOnSubmit();
            }

            if (!string.IsNullOrEmpty(bprInfo.edipiVerifiedDate))
            {
                registrant.REGISTRANT_RECORD_INFO.EDIPI_VERIFIED_DATE = DateTimeOffset.Parse(bprInfo.edipiVerifiedDate).UtcDateTime;
            }
            if (!string.IsNullOrEmpty(bprInfo.eulaAcceptedDate))
            {
                registrant.REGISTRANT_RECORD_INFO.EULA_ACCEPTED_DATE = DateTimeOffset.Parse(bprInfo.eulaAcceptedDate).UtcDateTime;
            }
            if (!string.IsNullOrEmpty(bprInfo.contactInfoVerifiedDate))
            {
                registrant.REGISTRANT_RECORD_INFO.CONTACT_INFO_VERIFIED_DATE = DateTimeOffset.Parse(bprInfo.contactInfoVerifiedDate).UtcDateTime;
            }
            registrant.REGISTRANT_RECORD_INFO.EDIPI_MISMATCHED_FLAG = bprInfo.edipiMismatched;
        }

        private static void AddSSNToRegistrant(BprInfo bprInfo, REGISTRANT registrant)
        {
            if (!string.IsNullOrEmpty(bprInfo.contactInfo.rssn))
            {
                registrant.Snum = bprInfo.contactInfo.rssn;
                registrant.UPDATEDBY = "REST Service";
                registrant.UPDATED = DateTime.Now;
                registrant.SetAsUpdateOnSubmit();
            }
        }

        private static void ProcessReconsideredDeploymentPhoneInfo(BprInfo bprInfo, REGISTRANT registrant)
        {
            if (bprInfo.deploymentReconsider.contact.phones.Length > 0 && bprInfo.contactInfo.phones.Length == 0)
            {
                // Registrant phone
                foreach (REGISTRANT_PHONE registrantPhone in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs)
                {
                    registrantPhone.SetAsDeleteOnSubmit();
                }
                if (bprInfo.deploymentReconsider.contact.phones.Length > 0)
                {
                    REGISTRANT_PHONE phone = new REGISTRANT_PHONE();
                    foreach (BprPhone phoneItem in bprInfo.deploymentReconsider.contact.phones)
                    {
                        if (!string.IsNullOrEmpty(phoneItem.number))
                        {
                            phone = new REGISTRANT_PHONE();
                            phone.REGISTRANT_ID = registrant.REGISTRANT_ID;
                            phone.PHONE_NUMBER = phoneItem.number;
                            phone.TYPE = phoneItem.type;
                            registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs.Add(phone);
                            registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs.Last().SetAsInsertOnSubmit();
                        }
                    }
                }
            }
        }

        private static void ProcessReconsideredDeploymentEmailInfo(BprInfo bprInfo, REGISTRANT registrant)
        {
            if (bprInfo.deploymentReconsider.contact.emails.Length > 0 && bprInfo.contactInfo.emails.Length == 0)
            {
                // Registrant email
                foreach (REGISTRANT_EMAIL registrantEamil in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs)
                {
                    registrantEamil.SetAsDeleteOnSubmit();
                }
                REGISTRANT_EMAIL email = new REGISTRANT_EMAIL();
                BprEmail emailItem = bprInfo.deploymentReconsider.contact.emails[0];
                if (!string.IsNullOrEmpty(emailItem.email))
                {
                    email = new REGISTRANT_EMAIL();
                    email.REGISTRANT_ID = registrant.REGISTRANT_ID;
                    email.EMAIL = emailItem.email;
                    email.TYPE = emailItem.type;
                    registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs.Add(email);
                    registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs.Last().SetAsInsertOnSubmit();
                }
            }
        }

        private static void ProcessReconsideredDeploymentAddressInfo(BprInfo bprInfo, REGISTRANT registrant)
        {
            //Reconsidered Deployment Contact Info:
            if (bprInfo.deploymentReconsider.contact.addresses.Length > 0 && bprInfo.contactInfo.addresses.Length == 0)
            {
                // Registrant Address Info
                foreach (REGISTRANT_ADDRESS registrantAddress in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ADDRESSes)
                {
                    registrantAddress.SetAsDeleteOnSubmit();
                }
                REGISTRANT_ADDRESS address = new REGISTRANT_ADDRESS();
                BprAddress addressItem = bprInfo.deploymentReconsider.contact.addresses[0];
                if (!string.IsNullOrEmpty(addressItem.street1))
                {
                    address = new REGISTRANT_ADDRESS();
                    address.REGISTRANT_ID = registrant.REGISTRANT_ID;
                    address.STREET1 = addressItem.street1;
                    address.STREET2 = addressItem.street2;
                    address.CITY = addressItem.city;
                    address.STATE = addressItem.state;
                    address.ZIP = addressItem.zip;
                    address.COUNTRY = addressItem.country;
                    registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ADDRESSes.Add(address);
                    registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ADDRESSes.Last().SetAsInsertOnSubmit();
                }
            }
        }

        private void ProcessReconsideredDeployments(BprInfo bprInfo, REGISTRANT registrant)
        {
            //Reconsidered Deployments:
            if (bprInfo.deploymentReconsider.history.Length > 0)
            {
                REGISTRANT_DEPLOYMENT reconsideredDeployment;
                foreach (BprDeploymentHistory bprDeploymentHistory in bprInfo.deploymentReconsider.history)
                {
                    if (!string.IsNullOrEmpty(bprDeploymentHistory.deploymentStartDate) && !string.IsNullOrEmpty(bprDeploymentHistory.deploymentEndDate))
                    {
                        #region reconsidered Deployments - check for updates
                        bool newDeployment, duplicate;
                        int branchOfServiceId;
                        CheckForUpdatedReconsideredDeployments(registrant, out reconsideredDeployment, bprDeploymentHistory, out newDeployment, out duplicate, out branchOfServiceId);
                        #endregion

                        #region reconsidered Deployments - insert or update
                        reconsideredDeployment = InsertOrUpdateReconsideredDeployments(registrant, reconsideredDeployment, bprDeploymentHistory, newDeployment, duplicate, branchOfServiceId);
                        #endregion
                    }
                }
            }
        }

        private REGISTRANT_DEPLOYMENT InsertOrUpdateReconsideredDeployments(REGISTRANT registrant, REGISTRANT_DEPLOYMENT reconsideredDeployment, BprDeploymentHistory bprDeploymentHistory, bool newDeployment, bool duplicate, int branchOfServiceId)
        {
            if (!duplicate)
            {
                var deploymentImporter = new BprDeploymentManager();
                if (newDeployment)
                {
                    reconsideredDeployment = new REGISTRANT_DEPLOYMENT();
                    reconsideredDeployment.STD_DEPLOYMENT_STAGE_ID = (int)Enums.DeploymentStage.NotSet;
                }
                reconsideredDeployment.REGISTRANT_ID = registrant.REGISTRANT_ID;
                reconsideredDeployment.DEPLOYMENT_BASE = bprDeploymentHistory.deploymentBase;
                deploymentImporter.ProcessDeploymentCountry(reconsideredDeployment, bprDeploymentHistory.deploymentCountry);

                reconsideredDeployment.DUTY_PROJECT_CODE = bprDeploymentHistory.dutyProjectCode;
                if (!string.IsNullOrEmpty(bprDeploymentHistory.deploymentStartDate))
                {
                    reconsideredDeployment.DEPLOYMENT_START_DATE = DateTimeOffset.Parse(bprDeploymentHistory.deploymentStartDate).UtcDateTime.Date;
                }
                if (!string.IsNullOrEmpty(bprDeploymentHistory.deploymentEndDate))
                {
                    reconsideredDeployment.DEPLOYMENT_END_DATE = DateTimeOffset.Parse(bprDeploymentHistory.deploymentEndDate).UtcDateTime.Date;
                }
                if (!string.IsNullOrEmpty(bprDeploymentHistory.lastUpdated))
                {
                    reconsideredDeployment.LAST_UPDATED_DATE = DateTimeOffset.Parse(bprDeploymentHistory.lastUpdated).UtcDateTime.Date;
                }
                reconsideredDeployment.USER_ENTERED_FLAG = bprDeploymentHistory.userEntered;
                reconsideredDeployment.USER_VERIFIED_FLAG = bprDeploymentHistory.userVerified;
                reconsideredDeployment.IS_ELIGIBLE_FLAG = bprDeploymentHistory.isEligible;
                reconsideredDeployment.IS_GULF_WAR_VETERAN_FLAG = bprDeploymentHistory.isGulfWarVeteran;
                reconsideredDeployment.PREVIOUS_VALUES = bprDeploymentHistory.previousValues;
                reconsideredDeployment.BPR_DEPLOYMENT_ID = bprDeploymentHistory._id;
                new RegistrantDeploymentUtilities().SetDeploymentStage(reconsideredDeployment, registrant.STD_REGISTRANT_STATUS_ID);
                if (!string.IsNullOrEmpty(bprDeploymentHistory.previousUpdated))
                {
                    reconsideredDeployment.PREVIOUS_UPDATED_DATE = DateTimeOffset.Parse(bprDeploymentHistory.previousUpdated).UtcDateTime.Date;
                }

                if (branchOfServiceId > 0)
                {
                    reconsideredDeployment.STD_BRANCH_OF_SERVICE_ID = branchOfServiceId;
                }
                String provenanceSource = "NA";
                if (bprDeploymentHistory.provenance != null &&
                    !String.IsNullOrEmpty(bprDeploymentHistory.provenance.source))
                {
                    provenanceSource = bprDeploymentHistory.provenance.source;
                }

                if (newDeployment)
                {
                    reconsideredDeployment.CREATEDBY = provenanceSource;
                    reconsideredDeployment.UPDATEDBY = provenanceSource;
                    registrant.REGISTRANT_DEPLOYMENTs.Add(reconsideredDeployment);
                    registrant.REGISTRANT_DEPLOYMENTs.Last().SetAsInsertOnSubmit();
                }
            }

            return reconsideredDeployment;
        }

        private void CheckForUpdatedReconsideredDeployments(REGISTRANT registrant, out REGISTRANT_DEPLOYMENT reconsideredDeployment, BprDeploymentHistory bprDeploymentHistory, out bool newDeployment, out bool duplicate, out int branchOfServiceId)
        {
            newDeployment = true;
            duplicate = false;
            branchOfServiceId = AHOBPRShared.GetStdBranchOfServiceId(bprDeploymentHistory.branchOfService);
            reconsideredDeployment = new REGISTRANT_DEPLOYMENT();
            reconsideredDeployment.STD_DEPLOYMENT_STAGE_ID = (int)Enums.DeploymentStage.NotSet;

            foreach (REGISTRANT_DEPLOYMENT sqlDeployment in registrant.REGISTRANT_DEPLOYMENTs)
            {
                if (!string.IsNullOrEmpty(bprDeploymentHistory.deploymentStartDate) && !string.IsNullOrEmpty(bprDeploymentHistory.deploymentEndDate))
                {

                    if (SqlDeploymentHasSameInfoAsBprDeployment(sqlDeployment, bprDeploymentHistory, branchOfServiceId))
                    {
                        if (sqlDeployment.LINQEntityState == EntityState.New)
                        {
                            duplicate = true;
                            break;
                        }
                        else if (sqlDeployment.LINQEntityState == EntityState.Deleted)
                        {
                            newDeployment = true;
                        }
                        else
                        {
                            newDeployment = false;
                            reconsideredDeployment = PopulateDeploymentsUpdatedAndCreatedFields(sqlDeployment);
                        }
                    }
                }
            }
        }
        /// <summary>
        /// Mandate from product owners to keep any existing registrant deployments. So merge any new data in and/or add to DB while keeping any old entries.
        /// </summary>
        /// <param name="bprInfo"></param>
        /// <param name="registrant"></param>
        private void MergeJsonDeploymentDataWithAnyExistingRegistrantDeploymentDatabaseEntries(BprInfo bprInfo, REGISTRANT registrant)
        {

            if (bprInfo.deploymentHistory.Length > 0)
            {
                var deployment = new REGISTRANT_DEPLOYMENT();
                var deploymentManager = new BprDeploymentManager();
                foreach (BprDeploymentHistory bprDeploymentItem in bprInfo.deploymentHistory)
                {
                    bprDeploymentItem.deploymentCountry = deploymentManager.ConvertDeploymentCountryCodeToName(bprDeploymentItem.deploymentCountry);
                    if (!string.IsNullOrEmpty(bprDeploymentItem.deploymentStartDate) && !string.IsNullOrEmpty(bprDeploymentItem.deploymentEndDate))
                    {
                        #region deployment history - check for updates
                        bool newDeployment = true;
                        bool duplicate = false;
                        int branchOfServiceId = AHOBPRShared.GetStdBranchOfServiceId(bprDeploymentItem.branchOfService);
                        CheckIfBprDeploymentExistsInDB(registrant, ref deployment, bprDeploymentItem, ref newDeployment, ref duplicate, branchOfServiceId);
                        #endregion deployment history - check for updates

                        #region deployment history - insert or update
                        if (!duplicate)
                        {
                            deployment = deploymentManager.ProcessDeployment(registrant, deployment, bprDeploymentItem, newDeployment, branchOfServiceId);
                        }
                        #endregion deployment history - insert or update
                    }
                }
            }
        }//end ProcessRegistrantDeployments  
        /// <summary>
        /// Compare the deployments in the REGISTRANT db object against the bprDeployment
        /// </summary>
        /// <param name="registrant"></param>
        /// <param name="deployment"></param>
        /// <param name="bprDeploymentItem"></param>
        /// <param name="newDeployment"></param>
        /// <param name="duplicate"></param>
        /// <param name="branchOfServiceId"></param>
        private static void CheckIfBprDeploymentExistsInDB(REGISTRANT registrant, ref REGISTRANT_DEPLOYMENT deployment, BprDeploymentHistory bprDeploymentItem, ref bool newDeployment, ref bool duplicate, int branchOfServiceId)
        {
                foreach (REGISTRANT_DEPLOYMENT sqlDeployment in registrant.REGISTRANT_DEPLOYMENTs)
                {
                    if (SqlDeploymentHasSameInfoAsBprDeployment(sqlDeployment, bprDeploymentItem, branchOfServiceId))
                    {
                        if (sqlDeployment.LINQEntityState == EntityState.New)
                        {
                            duplicate = true;
                        }
                        else if (sqlDeployment.LINQEntityState == EntityState.Deleted)
                        {
                            newDeployment = true;
                        }
                        else
                        {
                            newDeployment = false;
                            deployment = PopulateDeploymentsUpdatedAndCreatedFields(sqlDeployment);
                        }
                        break;
                    }

                }
        }
        private static REGISTRANT_DEPLOYMENT PopulateDeploymentsUpdatedAndCreatedFields(REGISTRANT_DEPLOYMENT sqlDeployment)
        {
            REGISTRANT_DEPLOYMENT deployment = sqlDeployment;
            deployment.UPDATEDBY = "REST Service";
            deployment.UPDATED = DateTime.Now;
            deployment.SetAsUpdateOnSubmit();
            return deployment;
        }
        static bool SqlDeploymentHasSameInfoAsBprDeployment(REGISTRANT_DEPLOYMENT sqlDeployment, BprDeploymentHistory bprDeploymentItem, int branchOfServiceId)
        {
            return (sqlDeployment.DEPLOYMENT_START_DATE.Value.Date == DateTimeOffset.Parse(bprDeploymentItem.deploymentStartDate).UtcDateTime.Date
                    && sqlDeployment.DEPLOYMENT_END_DATE.Value.Date == DateTimeOffset.Parse(bprDeploymentItem.deploymentEndDate).UtcDateTime.Date
                    && sqlDeployment.DEPLOYMENT_BASE == bprDeploymentItem.deploymentBase
                    && sqlDeployment.DEPLOYMENT_COUNTRY == bprDeploymentItem.deploymentCountry
                    && sqlDeployment.STD_BRANCH_OF_SERVICE_ID == branchOfServiceId);
        }
        private static void AddAddressToRegistrant(BprInfo bprInfo, REGISTRANT registrant)
        {
            // Registrant Address Info
            if (bprInfo.contactInfo.addresses.Length > 0)
            {
                REGISTRANT_ADDRESS address;
                BprAddress addressItem = bprInfo.contactInfo.addresses[0];
                //If there's an existing address, try to update it
                if (registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ADDRESSes.Count > 0)
                {
                    address = registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ADDRESSes[0];
                    if (address.STREET1 != addressItem.street1 ||
                            address.STREET2 != addressItem.street2 ||
                            address.CITY != addressItem.city ||
                            address.STATE != addressItem.state ||
                            address.ZIP != addressItem.zip ||
                            address.COUNTRY != addressItem.country)
                    {
                        address.STREET1 = addressItem.street1;
                        address.STREET2 = addressItem.street2;
                        address.CITY = addressItem.city;
                        address.STATE = addressItem.state;
                        address.ZIP = addressItem.zip;
                        address.COUNTRY = addressItem.country;

                        address.UPDATEDBY = "REST Service";
                        address.UPDATED = DateTime.Now;
                        address.SetAsUpdateOnSubmit();
                    }
                }
                else
                {
                    address = new REGISTRANT_ADDRESS();
                    if (!string.IsNullOrEmpty(addressItem.street1))
                    {
                        address = new REGISTRANT_ADDRESS();
                        address.REGISTRANT_ID = registrant.REGISTRANT_ID;
                        address.STREET1 = addressItem.street1;
                        address.STREET2 = addressItem.street2;
                        address.CITY = addressItem.city;
                        address.STATE = addressItem.state;
                        address.ZIP = addressItem.zip;
                        address.COUNTRY = addressItem.country;
                    }
                    registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_ADDRESSes.Add(address);
                    address.SetAsInsertOnSubmit();
                }
            }
        }

        private void ProcessServiceEpisodes(BprInfo bprInfo, REGISTRANT registrant)
        {
            // Service Episodes
            foreach (REGISTRANT_SERVICE_EPISODE registrantServiceEpisode in registrant.REGISTRANT_SERVICE_EPISODEs)
            {
                registrantServiceEpisode.SetAsDeleteOnSubmit();
            }

            if (bprInfo.serviceEpisodes.Length > 0)
            {
                ServiceEpisodeManager.UpdateRegistrantWithServiceEpisodes(bprInfo, registrant);
                SetRegistrantsCurrentComponentIdWithPersonnelCategoryCode(bprInfo, registrant);
            }
        }

        private void SetRegistrantsCurrentComponentIdWithPersonnelCategoryCode(BprInfo bprInfo, REGISTRANT registrant)
        {
            //Get the component from the latest service episode:
            BprServiceEpisode se = (from s in bprInfo.serviceEpisodes
                                    orderby s.serviceEpisodeEndDate descending
                                    select s).FirstOrDefault();
            //Set the registrant's current component ID:
            if (!String.IsNullOrEmpty(se.personnelCategoryCode))
            {
                registrant.STD_COMPONENT_ID_CURRENT = AHOBPRShared.GetStdComponent(se.personnelCategoryCode);
            }
            if (!String.IsNullOrEmpty(se.dischargeCharacterOfService))
            {
                registrant.STD_DISCHARGE_CHARACTER_ID = GetStdDischargeId(se.dischargeCharacterOfService);
            }
        }


        private static void AddPhoneNumberToRegistrant(BprInfo bprInfo, REGISTRANT registrant)
        {
            // Registrant phone
            //Update/Delete SQL Server phones:
            foreach (REGISTRANT_PHONE registrantPhone in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs)
            {
                var found = false;
                foreach (BprPhone phoneItem in bprInfo.contactInfo.phones)
                {
                    if (registrantPhone.TYPE == phoneItem.type)
                    {
                        found = true;
                        if (registrantPhone.PHONE_NUMBER != phoneItem.number)
                        {
                            registrantPhone.UPDATEDBY = "REST Service";
                            registrantPhone.UPDATED = DateTime.Now;
                            registrantPhone.PHONE_NUMBER = phoneItem.number;
                            registrantPhone.SetAsUpdateOnSubmit();
                        }
                    }
                }
                if (!found)
                {
                    registrantPhone.SetAsDeleteOnSubmit();
                }
            }

            //Add new JSON phones:                     
            foreach (BprPhone phoneItem in bprInfo.contactInfo.phones)
            {
                var found = false;
                foreach (REGISTRANT_PHONE registrantPhone in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs)
                {
                    found |= registrantPhone.TYPE == phoneItem.type;
                }

                if (!found)
                {
                    REGISTRANT_PHONE phone = new REGISTRANT_PHONE();
                    if (!string.IsNullOrEmpty(phoneItem.number))
                    {
                        phone = new REGISTRANT_PHONE();
                        phone.REGISTRANT_ID = registrant.REGISTRANT_ID;
                        phone.PHONE_NUMBER = phoneItem.number;
                        phone.TYPE = phoneItem.type;
                        registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs.Add(phone);
                        registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_PHONEs.Last().SetAsInsertOnSubmit();
                    }
                }
            }
        }

        private static void AddEmailAddressToRegistrant(BprInfo bprInfo, REGISTRANT registrant)
        {

            // Registrant email
            bool found;
            //Update/Delete SQL Server e-mails:
            foreach (REGISTRANT_EMAIL registrantEmail in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs)
            {
                found = false;
                foreach (BprEmail emailItem in bprInfo.contactInfo.emails)
                {
                    if (registrantEmail.TYPE == emailItem.type)
                    {
                        found = true;
                        if (registrantEmail.EMAIL != emailItem.email)
                        {
                            registrantEmail.UPDATEDBY = "REST Service";
                            registrantEmail.UPDATED = DateTime.Now;
                            registrantEmail.EMAIL = emailItem.email;
                            registrantEmail.SetAsUpdateOnSubmit();
                        }
                    }
                }
                if (!found)
                {
                    registrantEmail.SetAsDeleteOnSubmit();
                }
            }

            //Add new JSON e-mails:                        
            foreach (BprEmail emailItem in bprInfo.contactInfo.emails)
            {
                found = false;
                foreach (REGISTRANT_EMAIL registrantEmail in registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs)
                {
                    if (registrantEmail.TYPE == emailItem.type)
                    {
                        found = true;
                    }
                }

                if (!found)
                {
                    REGISTRANT_EMAIL email = new REGISTRANT_EMAIL();
                    if (!string.IsNullOrEmpty(emailItem.email))
                    {
                        email = new REGISTRANT_EMAIL();
                        email.REGISTRANT_ID = registrant.REGISTRANT_ID;
                        email.EMAIL = emailItem.email;
                        email.TYPE = emailItem.type;
                        registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs.Add(email);
                        registrant.REGISTRANT_CONTACT_INFO.REGISTRANT_EMAILs.Last().SetAsInsertOnSubmit();
                    }
                }
            }
        }

        private bool ParseInfoJsonDeploymentOnly(BprInfo bprInfo, List<string> deploymentIdsToBeAdded, List<string> allDeploymentIds)
        {
            bool result = false;
            try
            {
                if (bprInfo != null)
                {
                    REGISTRANT registrant = _registrantManager.GetRegistrantByUserIdWithInfoDeploymentOnly(bprInfo.userId);
                    if (registrant == null)
                    {
                        return result;
                    }
                    else
                    {
                        registrant.SetAsChangeTrackingRoot();
                        //Check to see if the deployments in deploymentIdsToBeAdded already exist in the user's deployments
                        foreach (string deploymentId in deploymentIdsToBeAdded)
                        {
                            bool theDeploymentFound = false;

                            #region deploymentHistory
                            if (bprInfo.deploymentHistory.Length > 0)
                            {
                                var deployment = new REGISTRANT_DEPLOYMENT();
                                deployment.STD_DEPLOYMENT_STAGE_ID = (int)Enums.DeploymentStage.New;

                                foreach (BprDeploymentHistory bprDeploymentItem in bprInfo.deploymentHistory)
                                {
                                    if (bprDeploymentItem._id == deploymentId)
                                    {
                                        if (!string.IsNullOrEmpty(bprDeploymentItem.deploymentStartDate) && !string.IsNullOrEmpty(bprDeploymentItem.deploymentEndDate))
                                        {
                                            int branchOfServiceId = AHOBPRShared.GetStdBranchOfServiceId(bprDeploymentItem.branchOfService);
                                            #region history - check if exists
                                            bool newDeployment = true;
                                            foreach (REGISTRANT_DEPLOYMENT sqlDeployment in registrant.REGISTRANT_DEPLOYMENTs)
                                            {

                                                if (SqlDeploymentHasSameInfoAsBprDeployment(sqlDeployment, bprDeploymentItem, branchOfServiceId))
                                                {
                                                    bool notUsed = true;
                                                    foreach (string responseDeploymentId in allDeploymentIds)
                                                    {
                                                        if (responseDeploymentId == sqlDeployment.BPR_DEPLOYMENT_ID)
                                                        {
                                                            notUsed = false;
                                                            break;
                                                        }
                                                    }
                                                    if (notUsed)
                                                    {
                                                        // Only do the update if the deployment ID in SQL table is not used in responses
                                                        newDeployment = false;
                                                        deployment = sqlDeployment;
                                                        deployment.UPDATED = DateTime.Now;
                                                        deployment.UPDATEDBY = "REST Service";
                                                        deployment.SetAsUpdateOnSubmit();
                                                    }
                                                    break;
                                                }
                                            }
                                            #endregion
                                            #region deployment history - update or insert
                                            deployment.REGISTRANT_ID = registrant.REGISTRANT_ID;
                                            deployment.DEPLOYMENT_BASE = bprDeploymentItem.deploymentBase;
                                            deployment.DEPLOYMENT_COUNTRY = bprDeploymentItem.deploymentCountry;
                                            deployment.DUTY_PROJECT_CODE = bprDeploymentItem.dutyProjectCode;
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.deploymentStartDate))
                                            {
                                                deployment.DEPLOYMENT_START_DATE = DateTimeOffset.Parse(bprDeploymentItem.deploymentStartDate).UtcDateTime.Date;
                                            }
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.deploymentEndDate))
                                            {
                                                deployment.DEPLOYMENT_END_DATE = DateTimeOffset.Parse(bprDeploymentItem.deploymentEndDate).UtcDateTime.Date;
                                            }
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.lastUpdated))
                                            {
                                                deployment.LAST_UPDATED_DATE = DateTimeOffset.Parse(bprDeploymentItem.lastUpdated).UtcDateTime.Date;
                                            }
                                            deployment.USER_ENTERED_FLAG = bprDeploymentItem.userEntered;
                                            deployment.USER_VERIFIED_FLAG = bprDeploymentItem.userVerified;
                                            deployment.IS_ELIGIBLE_FLAG = bprDeploymentItem.isEligible;
                                            deployment.IS_GULF_WAR_VETERAN_FLAG = bprDeploymentItem.isGulfWarVeteran;
                                            deployment.PREVIOUS_VALUES = bprDeploymentItem.previousValues;
                                            deployment.BPR_DEPLOYMENT_ID = bprDeploymentItem._id;
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.previousUpdated))
                                            {
                                                deployment.PREVIOUS_UPDATED_DATE = DateTimeOffset.Parse(bprDeploymentItem.previousUpdated).UtcDateTime;
                                            }

                                            if (branchOfServiceId > 0)
                                            {
                                                deployment.STD_BRANCH_OF_SERVICE_ID = branchOfServiceId;
                                            }
                                            if (!String.IsNullOrEmpty(bprDeploymentItem.personnelCategoryCode))
                                            {
                                                deployment.STD_COMPONENT_ID = AHOBPRShared.GetStdComponent(bprDeploymentItem.personnelCategoryCode);
                                            }

                                            String provenanceSource = "NA";
                                            if (bprDeploymentItem.provenance != null &&
                                                !String.IsNullOrEmpty(bprDeploymentItem.provenance.source))
                                            {
                                                provenanceSource = bprDeploymentItem.provenance.source;
                                            }
                                            deployment.CREATEDBY = provenanceSource;
                                            deployment.UPDATEDBY = provenanceSource;
                                            if (newDeployment)
                                            {
                                                registrant.REGISTRANT_DEPLOYMENTs.Add(deployment);
                                                registrant.REGISTRANT_DEPLOYMENTs.Last().SetAsInsertOnSubmit();
                                            }
                                            #endregion
                                            theDeploymentFound = true;
                                            break;
                                        }
                                    }
                                }
                            }
                            #endregion

                            #region deployment reconsider
                            if (theDeploymentFound == false)
                            {
                                //Reconsidered Deployments:
                                if (bprInfo.deploymentReconsider.history.Length > 0)
                                {
                                    REGISTRANT_DEPLOYMENT reconsideredDeployment = new REGISTRANT_DEPLOYMENT();
                                    reconsideredDeployment.STD_DEPLOYMENT_STAGE_ID = (int)Enums.DeploymentStage.New;

                                    foreach (BprDeploymentHistory bprDeploymentItem in bprInfo.deploymentReconsider.history)
                                    {
                                        if (bprDeploymentItem._id == deploymentId)
                                        {
                                            int branchOfServiceId = AHOBPRShared.GetStdBranchOfServiceId(bprDeploymentItem.branchOfService);
                                            bool newDeployment = true;
                                            #region deployment reconsider - check if exists
                                            foreach (REGISTRANT_DEPLOYMENT sqlDeployment in registrant.REGISTRANT_DEPLOYMENTs)
                                            {
                                                if (SqlDeploymentHasSameInfoAsBprDeployment(sqlDeployment, bprDeploymentItem, branchOfServiceId))
                                                {
                                                    bool notUsed = true;
                                                    foreach (string responseDeploymentId in allDeploymentIds)
                                                    {
                                                        if (responseDeploymentId == sqlDeployment.BPR_DEPLOYMENT_ID)
                                                        {
                                                            notUsed = false;
                                                            break;
                                                        }
                                                    }
                                                    if (notUsed)
                                                    {
                                                        // Only do the update if the deployment ID in SQL table is not used in responses
                                                        newDeployment = false;
                                                        reconsideredDeployment = sqlDeployment;
                                                        reconsideredDeployment.UPDATED = DateTime.Now;
                                                        reconsideredDeployment.UPDATEDBY = "REST Service";
                                                        reconsideredDeployment.SetAsUpdateOnSubmit();
                                                    }
                                                    break;
                                                }
                                            }
                                            #endregion
                                            #region deployment reconsider - update or insert
                                            reconsideredDeployment.REGISTRANT_ID = registrant.REGISTRANT_ID;
                                            reconsideredDeployment.DEPLOYMENT_BASE = bprDeploymentItem.deploymentBase;
                                            reconsideredDeployment.DEPLOYMENT_COUNTRY = bprDeploymentItem.deploymentCountry;
                                            reconsideredDeployment.DUTY_PROJECT_CODE = bprDeploymentItem.dutyProjectCode;
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.deploymentStartDate))
                                            {
                                                reconsideredDeployment.DEPLOYMENT_START_DATE = DateTimeOffset.Parse(bprDeploymentItem.deploymentStartDate).UtcDateTime.Date;
                                            }
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.deploymentEndDate))
                                            {
                                                reconsideredDeployment.DEPLOYMENT_END_DATE = DateTimeOffset.Parse(bprDeploymentItem.deploymentEndDate).UtcDateTime.Date;
                                            }
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.lastUpdated))
                                            {
                                                reconsideredDeployment.LAST_UPDATED_DATE = DateTimeOffset.Parse(bprDeploymentItem.lastUpdated).UtcDateTime.Date;
                                            }
                                            reconsideredDeployment.USER_ENTERED_FLAG = bprDeploymentItem.userEntered;
                                            reconsideredDeployment.USER_VERIFIED_FLAG = bprDeploymentItem.userVerified;
                                            reconsideredDeployment.IS_ELIGIBLE_FLAG = bprDeploymentItem.isEligible;
                                            reconsideredDeployment.IS_GULF_WAR_VETERAN_FLAG = bprDeploymentItem.isGulfWarVeteran;
                                            reconsideredDeployment.PREVIOUS_VALUES = bprDeploymentItem.previousValues;
                                            reconsideredDeployment.BPR_DEPLOYMENT_ID = bprDeploymentItem._id;
                                            if (!string.IsNullOrEmpty(bprDeploymentItem.previousUpdated))
                                            {
                                                reconsideredDeployment.PREVIOUS_UPDATED_DATE = DateTimeOffset.Parse(bprDeploymentItem.previousUpdated).UtcDateTime.Date;
                                            }

                                            if (branchOfServiceId > 0)
                                            {
                                                reconsideredDeployment.STD_BRANCH_OF_SERVICE_ID = branchOfServiceId;
                                            }
                                            String provenanceSource = "NA";
                                            if (bprDeploymentItem.provenance != null &&
                                                !String.IsNullOrEmpty(bprDeploymentItem.provenance.source))
                                            {
                                                provenanceSource = bprDeploymentItem.provenance.source;
                                            }

                                            reconsideredDeployment.CREATEDBY = provenanceSource;
                                            reconsideredDeployment.UPDATEDBY = provenanceSource;
                                            if (newDeployment)
                                            {
                                                registrant.REGISTRANT_DEPLOYMENTs.Add(reconsideredDeployment);
                                                registrant.REGISTRANT_DEPLOYMENTs.Last().SetAsInsertOnSubmit();
                                            }
                                            #endregion
                                            break;
                                        }
                                    }
                                }
                            }
                            #endregion
                        }
                        // Save Registrant Info
                        result = (_registrantManager.UpdateRegistrant(registrant) >= 0);
                    }
                }

            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Parse JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "Error: " + ex.Message);
                result = false;
            }

            return result;
        }

        //private int GetStdServiceOccupation(String serviceOccupation)
        //{
        //    return (new AHOBPRGenericManager()).GetStdServiceOccupationId(serviceOccupation);
        //}

        private int GetStdDischargeId(String discharge)
        {
            return (new AHOBPRGenericManager()).GetStdDischargeCharacterId(discharge);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="jsonData"></param>
        /// <returns></returns>
        private bool ParseResponseJsonFollowup(string jsonData, JSON_QUEUE_FOLLOWUP queueFollowup)
        {
            bool result = false;
            JSON_QUEUE_FOLLOWUP_STATUS queueStatus = new JSON_QUEUE_FOLLOWUP_STATUS();
            queueStatus.JSON_QUEUE_FOLLOWUP_ID = queueFollowup.JSON_QUEUE_FOLLOWUP_ID;
            queueStatus.REGISTRANT_JSON_FOLLOWUP_ID = queueFollowup.REGISTRANT_JSON_FOLLOWUP_ID;
            queueStatus.PROCESS_START_DATE = DateTime.Now;
            queueStatus.PROCESS_COMPLETE_DATE = DateTime.Now;
            try
            {
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                serializer.RegisterConverters(new[] { new AnswerJavaScriptConverter() });
                BprResponses bprResponses = serializer.Deserialize<BprResponses>(jsonData);
                if (bprResponses != null)
                {
                    REGISTRANT registrant = _registrantManager.GetRegistrantByUserIdWithFollowupResponse(bprResponses.userId);
                    if (registrant == null)
                    {
                        #region registrant not found
                        queueStatus.PROCESS_COMPLETE_DATE = DateTime.Now;
                        queueStatus.PROCESS_STATUS = "failed";
                        queueStatus.PROCESS_ERROR = "Registrant not found";
                        SaveJsonQueueFollowupData(queueStatus);
                        return false;
                        #endregion
                    }
                    else
                    {
                        #region registrant found, continue ....
                        registrant.SetAsChangeTrackingRoot();
                        int formId = _formManager.GetFollowupFormId(bprResponses.formId);

                        if (formId == 0)
                        {
                            queueStatus.PROCESS_COMPLETE_DATE = DateTime.Now;
                            queueStatus.PROCESS_STATUS = "failed";
                            queueStatus.PROCESS_ERROR = "Form record not found";
                            SaveJsonQueueFollowupData(queueStatus);
                            return false;
                        }
                        else
                        {
                            // Form Response Record
                            //Wrong: Take all e.STD_FORM.BPR_FORM_IDs that match and use the first of them
                            //FOLLOWUP_FORM_RESPONSE response = (from e in registrant.FOLLOWUP_FORM_RESPONSEs
                            //                                   where e.STD_FORM.BPR_FORM_ID == bprResponses.formId
                            //                                   select e).FirstOrDefault();
                            //Correct: Get and use the first e.STD_FORM.BPR_FORM_ID that matches the formId

                            FOLLOWUP_FORM_RESPONSE response = GetFirstFollowupFormResponse(bprResponses, registrant);
                            if (response == null)
                            {
                                if (_formManager.IsFormSystemWide(formId))
                                {
                                    bool exists = false;
                                    _registrantManager.AddFollowupQuestionsToOneReigstrant(registrant.REGISTRANT_ID, formId, ref exists);
                                    registrant = _registrantManager.GetRegistrantByUserIdWithFollowupResponse(bprResponses.userId);
                                    registrant.SetAsChangeTrackingRoot();

                                    response = GetFirstFollowupFormResponse(bprResponses, registrant);
                                }
                                else
                                {
                                    #region FOLLOWUP_FORM_RESPONSE record not found, exit
                                    queueStatus.PROCESS_STATUS = "failed";
                                    queueStatus.PROCESS_ERROR = "The form response record is missing in FOLLOWUP_FORM_RESPONSE table";
                                    SaveJsonQueueFollowupData(queueStatus);
                                    return false;
                                    #endregion
                                }
                            }

                            #region FOLLOWUP_FORM_RESPONSE record found. Continue...
                            response.UPDATED = DateTime.Now;
                            response.UPDATEDBY = "REST Service";
                            response.SetAsUpdateOnSubmit();

                            // Process questions and answers data only when the questionnaire is completed
                            if (!string.IsNullOrEmpty(bprResponses.responseStatus.questionnaireCompletedDate))
                            {
                                response.COMPLETED_DATE = DateTimeOffset.Parse(bprResponses.responseStatus.questionnaireCompletedDate).UtcDateTime;

                                #region delete previous questions & answers
                                // Delete the previous value
                                foreach (FOLLOWUP_FORM_RESPONSE_QUESTION question in response.FOLLOWUP_FORM_RESPONSE_QUESTIONs)
                                {
                                    question.SetAsDeleteOnSubmit();
                                    foreach (FOLLOWUP_FORM_RESPONSE_ANSWER answer in question.FOLLOWUP_FORM_RESPONSE_ANSWERs)
                                    {
                                        answer.SetAsDeleteOnSubmit();
                                    }
                                }
                                #endregion

                                #region insert new values
                                // Insert the new values
                                FOLLOWUP_FORM_RESPONSE_QUESTION sqlQuestion = new FOLLOWUP_FORM_RESPONSE_QUESTION();
                                foreach (BprResponse bprResponse in bprResponses.responses)
                                {
                                    bool hasAnswer = false;
                                    if (bprResponse.answers.Length > 0)
                                    {
                                        hasAnswer |= !string.IsNullOrEmpty(bprResponse.answers[0].answerId);
                                    }

                                    if (!string.IsNullOrEmpty(bprResponse.sectionId) && !string.IsNullOrEmpty(bprResponse.subSectionId)
                                        && !string.IsNullOrEmpty(bprResponse.questionId) && hasAnswer)
                                    {
                                        sqlQuestion = new FOLLOWUP_FORM_RESPONSE_QUESTION();
                                        sqlQuestion.FOLLOWUP_FORM_RESPONSE_ID = response.FOLLOWUP_FORM_RESPONSE_ID;
                                        sqlQuestion.STARTED_DATE = ParseGmtDateTime(bprResponses.responseStatus.questionnaireCompletedDate);
                                        sqlQuestion.COMPLETED_DATE = ParseGmtDateTime(bprResponses.responseStatus.questionnaireCompletedDate);
                                        int sqlQuestionId = _formManager.GetFormQuestionIdForFollowupQuestion(formId, bprResponse.questionId);
                                        if (sqlQuestionId > 0)
                                        {
                                            sqlQuestion.STD_FORM_QUESTION_ID = sqlQuestionId;
                                            int deploymentId = _formManager.GetDeploymentId(registrant.REGISTRANT_ID, bprResponse.deploymentId);
                                            if (bprResponse.answers.Length > 0)
                                            {
                                                response.FOLLOWUP_FORM_RESPONSE_QUESTIONs.Add(sqlQuestion);
                                                sqlQuestion.SetAsInsertOnSubmit();

                                                foreach (BprAnswer bprAnswer in bprResponse.answers)
                                                {
                                                    FOLLOWUP_FORM_RESPONSE_ANSWER sqlAnswer = new FOLLOWUP_FORM_RESPONSE_ANSWER();
                                                    sqlAnswer.FOLLOWUP_FORM_RESPONSE_QUESTION_ID = sqlQuestion.FOLLOWUP_FORM_RESPONSE_QUESTION_ID;
                                                    if (!string.IsNullOrEmpty(bprAnswer.answerValue))
                                                    {
                                                        sqlAnswer.ANSWER_VALUE = bprAnswer.answerValue;
                                                    }
                                                    int sqlAnswerId = _formManager.GetFormAnswerId(sqlQuestionId, bprAnswer.answerId);
                                                    if (sqlAnswerId > 0)
                                                    {
                                                        sqlAnswer.STD_FORM_ANSWER_ID = sqlAnswerId;
                                                        sqlQuestion.FOLLOWUP_FORM_RESPONSE_ANSWERs.Add(sqlAnswer);
                                                        sqlAnswer.SetAsInsertOnSubmit();
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }

                                #endregion

                                #region save registrant
                                // Save to SQL database
                                result = (_registrantManager.UpdateRegistrant(registrant) >= 0);

                                if (result)
                                {
                                    queueStatus.PROCESS_STATUS = "success";
                                    SaveJsonQueueFollowupData(queueStatus);
                                }
                                else
                                {
                                    queueStatus.PROCESS_STATUS = "failed";
                                    queueStatus.PROCESS_ERROR = "failed to save to database";
                                    SaveJsonQueueFollowupData(queueStatus);
                                }
                                #endregion
                            }
                            #endregion

                        }
                    }
                    #endregion
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Parse JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "JSON QueueFollowupId: " + queueStatus.JSON_QUEUE_FOLLOWUP_ID.ToString() + ", Error: " + ex.Message);
                queueStatus.PROCESS_COMPLETE_DATE = DateTime.Now;
                queueStatus.PROCESS_STATUS = "failed";
                queueStatus.PROCESS_ERROR = ex.Message;
                SaveJsonQueueFollowupData(queueStatus);
                result = false;
            }

            return result;
        }

        private static FOLLOWUP_FORM_RESPONSE GetFirstFollowupFormResponse(BprResponses bprResponses, REGISTRANT registrant)
        {
            return registrant.FOLLOWUP_FORM_RESPONSEs.FirstOrDefault(e => e.STD_FORM.BPR_FORM_ID == bprResponses.formId);
        }

        /// <summary>
        /// Update user json based on response status
        /// If the response has questionnaire completed date, change the user status to "PARTICIPANT" if it not true
        /// Else if the response has questionnaire started date, change the user status to "ELIGIBLE" if it not in "ELIGIBLE" or "REVIEWED_ELIGIBLE" status
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="bprResponseStatus"></param>
        private void UpdateUserStatusFromResponseStatus(string userId, BprResponseStatus bprResponseStatus)
        {
            bool statusChanged = false;
            JavaScriptSerializer serializer = new JavaScriptSerializer();
            try
            {
                REGISTRANT_JSON record = GetJsonData(userId, AHOBPRGlobal.AhobprJsonDataTypeUser);
                if (record != null)
                {
                    BprUser bprUser = serializer.Deserialize<BprUser>(record.JSON_DATA);
                    if (bprUser != null && bprResponseStatus != null)
                    {
                        if (!string.IsNullOrEmpty(bprResponseStatus.questionnaireCompletedDate))
                        {
                            if (bprUser.userState.state != "PARTICIPANT")
                            {
                                bprUser.userState.state = "PARTICIPANT";
                                statusChanged = true;
                            }
                        }
                        else if (!string.IsNullOrEmpty(bprResponseStatus.questionnaireStartedDate))
                        {
                            if (bprUser.userState.state != "REVIEWED_ELIGIBLE" && bprUser.userState.state != "ELIGIBLE")
                            {
                                bprUser.userState.state = "ELIGIBLE";
                                statusChanged = true;
                            }
                        }
                        if (statusChanged)
                        {
                            SaveJsonData(userId, AHOBPRGlobal.AhobprJsonDataTypeUser, serializer.Serialize(bprUser), string.Empty);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Update User Json Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "Error: " + ex.Message);
            }
        }

        /// <summary>
        /// Parse Response Json
        /// </summary>
        /// <param name="jsonData"></param>
        /// <returns></returns>
        private bool ParseResponseJson(string jsonData, JSON_QUEUE queue)
        {
            bool result = false;
            queue.SetAsChangeTrackingRoot();
            queue.SetAsUpdateOnSubmit();
            queue.PROCESS_START_DATE = DateTime.Now;
            try
            {

                JavaScriptSerializer serializer = new JavaScriptSerializer();
                serializer.RegisterConverters(new[] { new AnswerJavaScriptConverter() });
                BprResponses bprResponses = serializer.Deserialize<BprResponses>(jsonData);
                if (bprResponses != null && !string.IsNullOrEmpty(bprResponses.responseStatus.questionnaireStartedDate))
                {
                    // Check user json for status.  Update the status as needed
                    UpdateUserStatusFromResponseStatus(bprResponses.userId, bprResponses.responseStatus);

                    // Parse responses data
                    String contactPreference = "";
                    REGISTRANT registrant = _registrantManager.GetRegistrantByUserIdWithResponse(bprResponses.userId);
                    if (registrant == null)
                    {
                        MarkQueueEntryAsErrored(queue, "Registrant not found");
                        result = false;
                        return result;
                    }
                    else
                    {
                        registrant.SetAsChangeTrackingRoot();

                        int formId = _formManager.GetFormId(bprResponses.formId);

                        if (formId == 0)
                        {
                            MarkQueueEntryAsErrored(queue, "Form record not found");
                            result = false;
                            return result;
                        }
                        else
                        {
                            bool responseRecordExist = false;

                            // Form Response Record
                            foreach (FORM_RESPONSE response in registrant.FORM_RESPONSEs)
                            {
                                if (response.STD_FORM_ID == formId)
                                {
                                    responseRecordExist = true;
                                    break;
                                }
                            }
                            if (responseRecordExist == false)
                            {
                                FORM_RESPONSE formResponse = new FORM_RESPONSE();
                                formResponse.REGISTRANT_ID = registrant.REGISTRANT_ID;
                                formResponse.STD_FORM_ID = formId;

                                registrant.FORM_RESPONSEs.Add(formResponse);
                                registrant.FORM_RESPONSEs.Last().SetAsInsertOnSubmit();
                            }

                            // Form Response Status Record
                            foreach (FORM_RESPONSE response in registrant.FORM_RESPONSEs)
                            {
                                if (response.STD_FORM_ID == formId)
                                {
                                    if (response.FORM_RESPONSE_STATUS.Count > 0)
                                    {
                                        response.FORM_RESPONSE_STATUS.Last().UPDATED = DateTime.Now;
                                        response.FORM_RESPONSE_STATUS.Last().UPDATEDBY = "REST Service";
                                        response.FORM_RESPONSE_STATUS.Last().SetAsUpdateOnSubmit();
                                    }
                                    else
                                    {
                                        FORM_RESPONSE_STATUS status = new FORM_RESPONSE_STATUS();
                                        response.FORM_RESPONSE_STATUS.Add(status);
                                        response.FORM_RESPONSE_STATUS.Last().SetAsInsertOnSubmit();
                                    }
                                    response.FORM_RESPONSE_STATUS.Last().FORM_RESPONSE_ID = response.FORM_RESPONSE_ID;
                                    if (!string.IsNullOrEmpty(bprResponses.responseStatus.questionnaireStartedDate))
                                    {
                                        response.FORM_RESPONSE_STATUS.Last().QUESTIONNAIRE_STARTED_DATE = DateTimeOffset.Parse(bprResponses.responseStatus.questionnaireStartedDate).UtcDateTime;
                                    }
                                    if (!string.IsNullOrEmpty(bprResponses.responseStatus.questionnaireCompletedDate))
                                    {
                                        response.FORM_RESPONSE_STATUS.Last().QUESTIONNAIRE_COMPLETED_DATE = DateTimeOffset.Parse(bprResponses.responseStatus.questionnaireCompletedDate).UtcDateTime;
                                    }

                                    if (!string.IsNullOrEmpty(bprResponses.responseStatus.lastSectionId))
                                    {
                                        int sectionId = _formManager.GetSectionId(formId, Convert.ToInt32(bprResponses.responseStatus.lastSectionId));
                                        response.FORM_RESPONSE_STATUS.Last().LAST_SECTION_ID = sectionId;

                                        if (!string.IsNullOrEmpty(bprResponses.responseStatus.lastSubSectionId))
                                        {
                                            response.FORM_RESPONSE_STATUS.Last().LAST_SUB_SECTION_ID = _formManager.GetSubSectionId(formId, sectionId, Convert.ToInt32(bprResponses.responseStatus.lastSubSectionId));
                                        }
                                    }

                                    // Process questions and answers data only when the questionnaire is completed
                                    if (!string.IsNullOrEmpty(bprResponses.responseStatus.questionnaireCompletedDate))
                                    {
                                        // Make sure all the deployments exists in the REGISTRANT_DEPLOYMENT table.
                                        // If not, reprocess the info object to find the the deployment records.
                                        // if the deployments cannot be found in the info object, then fail.
                                        if (AllDeploymentsExist(queue.REGISTRANT_JSON_ID, jsonData, registrant.REGISTRANT_ID))
                                        {


                                            // Delete the previous value
                                            foreach (FORM_RESPONSE_QUESTION question in response.FORM_RESPONSE_QUESTIONs)
                                            {
                                                question.SetAsDeleteOnSubmit();
                                                foreach (FORM_RESPONSE_ANSWER answer in question.FORM_RESPONSE_ANSWERs)
                                                {
                                                    answer.SetAsDeleteOnSubmit();
                                                }
                                            }

                                            // Insert the new values
                                            FORM_RESPONSE_QUESTION sqlQuestion = new FORM_RESPONSE_QUESTION();
                                            foreach (BprResponse bprResponse in bprResponses.responses)
                                            {
                                                bool hasAnswer = false;
                                                if (bprResponse.answers.Length > 0)
                                                {
                                                    if (!string.IsNullOrEmpty(bprResponse.answers[0].answerId))
                                                    {
                                                        hasAnswer = true;
                                                    }
                                                }

                                                if (!string.IsNullOrEmpty(bprResponse.sectionId) && !string.IsNullOrEmpty(bprResponse.subSectionId)
                                                    && !string.IsNullOrEmpty(bprResponse.questionId) && hasAnswer)
                                                {
                                                    sqlQuestion = new FORM_RESPONSE_QUESTION();
                                                    sqlQuestion.FORM_RESPONSE_ID = response.FORM_RESPONSE_ID;
                                                    if (!string.IsNullOrEmpty(bprResponse.startedDate) && ParseGmtDateTime(bprResponse.startedDate) != DateTime.MinValue)
                                                    {
                                                        sqlQuestion.STARTED_DATE = ParseGmtDateTime(bprResponse.startedDate);
                                                    }
                                                    if (!string.IsNullOrEmpty(bprResponse.completedDate) && ParseGmtDateTime(bprResponse.completedDate) != DateTime.MinValue)
                                                    {
                                                        sqlQuestion.COMPLETED_DATE = ParseGmtDateTime(bprResponse.completedDate);
                                                    }

                                                    int sqlQuestionId = _formManager.GetFormQuestionId(formId, Convert.ToInt32(bprResponse.sectionId), Convert.ToInt32(bprResponse.subSectionId), bprResponse.questionId);
                                                    if (sqlQuestionId > 0)
                                                    {
                                                        sqlQuestion.STD_FORM_QUESTION_ID = sqlQuestionId;
                                                        int deploymentId = _formManager.GetDeploymentId(registrant.REGISTRANT_ID, bprResponse.deploymentId);
                                                        if (deploymentId > 0)
                                                        {
                                                            sqlQuestion.REGISTRANT_DEPLOYMENT_ID = deploymentId;
                                                        }
                                                        else if (!string.IsNullOrEmpty(bprResponse.deploymentId))
                                                        {
                                                            MarkQueueEntryAsErrored(queue, "Deployment history cannot be found");
                                                            return false;
                                                        }

                                                        if (bprResponse.answers.Length > 0)
                                                        {
                                                            //If there isn't already a form response question for the same question, 
                                                            //then add it:

                                                            bool formQuestionAlreadyExists = false;
                                                            foreach (FORM_RESPONSE_QUESTION q in response.FORM_RESPONSE_QUESTIONs)
                                                            {
                                                                if (q.FORM_RESPONSE_ID == sqlQuestion.FORM_RESPONSE_ID &&
                                                                    q.STD_FORM_QUESTION_ID == sqlQuestion.STD_FORM_QUESTION_ID
                                                                           & q.REGISTRANT_DEPLOYMENT_ID == sqlQuestion.REGISTRANT_DEPLOYMENT_ID
                                                                    && q.LINQEntityState != EntityState.Deleted)
                                                                {
                                                                    formQuestionAlreadyExists = true;
                                                                    break;
                                                                }
                                                            }

                                                            if (!formQuestionAlreadyExists)
                                                            {
                                                                response.FORM_RESPONSE_QUESTIONs.Add(sqlQuestion);
                                                                response.FORM_RESPONSE_QUESTIONs.Last().SetAsInsertOnSubmit();
                                                                foreach (BprAnswer bprAnswer in bprResponse.answers)
                                                                {
                                                                    FORM_RESPONSE_ANSWER sqlAnswer = new FORM_RESPONSE_ANSWER();
                                                                    sqlAnswer.FORM_RESPONSE_QUESTION_ID = response.FORM_RESPONSE_QUESTIONs.Last().FORM_RESPONSE_QUESTION_ID;
                                                                    if (!string.IsNullOrEmpty(bprAnswer.answerValue))
                                                                    {
                                                                        sqlAnswer.ANSWER_VALUE = bprAnswer.answerValue;

                                                                        // for weight
                                                                        if (bprResponse.sectionId == "2" && bprResponse.subSectionId == "3" && bprResponse.questionId == "B")
                                                                        {
                                                                            sqlAnswer.ANSWER_VALUE = sqlAnswer.ANSWER_VALUE + " pounds";
                                                                        }
                                                                    }
                                                                    else if (bprAnswer.answerValues.Length > 0)
                                                                    {
                                                                        string height = string.Empty;
                                                                        int index = 0;
                                                                        foreach (string value in bprAnswer.answerValues)
                                                                        {
                                                                            if (!string.IsNullOrEmpty(value))
                                                                            {
                                                                                height = height + value + (index == 0 ? " feet " : " inches");
                                                                            }
                                                                            index++;
                                                                        }
                                                                        if (!string.IsNullOrEmpty(height))
                                                                        {
                                                                            sqlAnswer.ANSWER_VALUE = height;
                                                                        }
                                                                    }
                                                                    int sqlAnswerId = _formManager.GetFormAnswerId(sqlQuestionId, bprAnswer.answerId);
                                                                    if (sqlAnswerId > 0)
                                                                    {
                                                                        sqlAnswer.STD_FORM_ANSWER_ID = sqlAnswerId;
                                                                        response.FORM_RESPONSE_QUESTIONs.Last().FORM_RESPONSE_ANSWERs.Add(sqlAnswer);
                                                                        response.FORM_RESPONSE_QUESTIONs.Last().FORM_RESPONSE_ANSWERs.Last().SetAsInsertOnSubmit();

                                                                        //for contact preference:
                                                                        if (bprResponse.sectionId == "8" &&
                                                                            bprResponse.subSectionId == "1" &&
                                                                            bprResponse.questionId == "A")
                                                                        {
                                                                            contactPreference = _formManager.GetFormAnswer(sqlQuestionId, bprAnswer.answerId);
                                                                        }
                                                                    }
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                        else
                                        {
                                            _sharedManager.LogErrorMessage("Parse JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "JSON QueueId: " + queue.JSON_QUEUE_ID.ToString() + ", Error: Not all deployments can be found.");
                                            MarkQueueEntryAsErrored(queue, "Not all deployments can be found");
                                            return false;
                                        }
                                    }
                                    break;
                                }
                            }

                        }

                    }

                    //If a contact preference was set, make sure it's in AHOBPR.REGISTRANT.CONTACT_PREFERENCE
                    if (!String.IsNullOrEmpty(contactPreference))
                    {
                        registrant.CONTACT_PREFERENCE = contactPreference;
                    }

                    // Save to SQL database
                    result = (_registrantManager.UpdateRegistrant(registrant) >= 0);

                    //If a contact preference was set, modify any info JSON with the preference.
                    if (!String.IsNullOrEmpty(contactPreference))
                    {
                        UpdateInfoJsonForContactInfo(registrant.USER_ID, contactPreference);
                    }

                    queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                    if (result)
                    {
                        queue.PROCESS_STATUS = "success";
                        SaveJsonQueueData(queue);
                        _formManager.UpdateRegistrantWithResponseInfo(registrant.REGISTRANT_ID);
                        //insert into bens queue 
                        using (var db = GetLocalContext(_connectionInfo))
                        {
                            db.SP_InsertIntoBensUpdates(registrant.EDIPI);
                        }

                    }
                    else
                    {
                        MarkQueueEntryAsErrored(queue, "failed to save to database");
                        SaveJsonData(registrant.USER_ID, AHOBPRGlobal.AhobprJsonDataTypeResponses, jsonData, "Failed Save");
                    }

                }
                else   // No need to process the response JSON data if questionnaire start date is null
                {
                    queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                    queue.PROCESS_STATUS = "success";
                    SaveJsonQueueData(queue);
                    result = true;
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Parse JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, "JasonQueueId: " + queue.JSON_QUEUE_ID.ToString() + ", Error: " + ex.Message);
                MarkQueueEntryAsErrored(queue, ex.Message);
                SaveJsonQueueData(queue);
                result = false;
            }

            return result;
        }

        private void MarkQueueEntryAsErrored(JSON_QUEUE queue, string errorMessage)
        {
            queue.PROCESS_COMPLETE_DATE = DateTime.Now;
            queue.PROCESS_STATUS = "failed";
            queue.PROCESS_ERROR = errorMessage;
            SaveJsonQueueData(queue);
        }

        public List<string> GetDeploymentIdsFromResponseObject(string responseJson)
        {
            List<string> deploymentIds = new List<string>();
            JavaScriptSerializer serializer = new JavaScriptSerializer();
            serializer.RegisterConverters(new[] { new AnswerJavaScriptConverter() });
            BprResponses bprResponses = serializer.Deserialize<BprResponses>(responseJson);
            string deploymentId = string.Empty;
            foreach (BprResponse bprResponse in bprResponses.responses)
            {
                if (!string.IsNullOrEmpty(bprResponse.deploymentId))
                {
                    if (bprResponse.deploymentId != deploymentId)
                    {
                        deploymentIds.Add(bprResponse.deploymentId);
                        deploymentId = bprResponse.deploymentId;
                    }
                }
            }
            return deploymentIds;
        }

        /// <summary>
        /// Check if all the deployment IDs used in the response object is in REGISTRANT_DEPLOYMENT table already.
        /// If not all exists, re-process info object deployment sections and only add the missings ones.
        /// If not all all deployments can be found in the info object, return false
        /// The reason for this is that the deployment might be missed for the first round of parsing due to duplicates.
        /// </summary>
        /// <param name="registrantJsonId"></param>
        /// <param name="responseJson"></param>
        /// <param name="registrantId"></param>
        /// <returns></returns>
        private bool AllDeploymentsExist(int registrantJsonId, string responseJson, int registrantId)
        {
            bool result = false;

            JavaScriptSerializer serializer = new JavaScriptSerializer();
            List<string> deploymentIds = GetDeploymentIdsFromResponseObject(responseJson);
            List<string> sqlDeploymentIds = _formManager.GetAllBprDeploymentIdsbyRegistrantId(registrantId);
            List<string> deploymentIdsToBeAdded = new List<string>();

            // Check if all deployments already exists in SQL DB
            foreach (string deploymentId in deploymentIds)
            {
                bool notFound = true;
                foreach (string sqlId in sqlDeploymentIds)
                {
                    if (sqlId == deploymentId)
                    {
                        notFound = false;
                        break;
                    }
                }
                if (notFound)
                {
                    deploymentIdsToBeAdded.Add(deploymentId);
                }
            }

            result = (deploymentIdsToBeAdded.Count == 0);

            // If not all deployments already exists in SQL DB, try to get them from info object
            if (result == false)
            {
                BprInfo bprInfo = GetInfoJsonByResponseId(registrantJsonId);
                if (bprInfo != null)
                {
                    result = ParseInfoJsonDeploymentOnly(bprInfo, deploymentIdsToBeAdded, deploymentIds);
                }
            }

            return result;
        }

        public bool UpdateInfoJsonForContactInfo(String userId, String contactPreference)
        {
            try
            {
                REGISTRANT_JSON record = GetJsonData(userId, AHOBPRGlobal.AhobprJsonDataTypeInfo);
                if (record != null)
                {
                    JavaScriptSerializer serializer = new JavaScriptSerializer();
                    BprInfo bprInfo = serializer.Deserialize<BprInfo>(record.JSON_DATA);
                    bprInfo.contactInfo.contactPreference = contactPreference;
                    SaveJsonData(record.OBJECT_ID, record.OBJECT_TYPE, serializer.Serialize(bprInfo), string.Empty);
                }
                return true;
            }
            catch (Exception ex)
            {
                AHOBPRLogger.LogErrorMessage("Exception: " + ex.Message, "AHOBPRJsonDataManager." + MethodBase.GetCurrentMethod().Name, ex.StackTrace);
                return false;
            }
        }

        private DateTime ParseGmtDateTime(string text)
        {
            try
            {
                int gmtIndex = text.IndexOf(" GMT", StringComparison.InvariantCulture);
                int zoneIndex = text.IndexOf(" (", StringComparison.InvariantCulture);

                string localText = text.Substring(0, gmtIndex);
                string offsetText = text.Substring(gmtIndex, zoneIndex - gmtIndex).Replace("GMT", "");

                string format = "ddd MMM d yyyy HH:mm:ss";
                DateTime newDate = DateTime.ParseExact(localText, format, new CultureInfo("en-US"));
                newDate = newDate.AddHours(Convert.ToInt16(offsetText));
                return newDate;
            }
            catch
            {
                return DateTime.MinValue;
            }
        }

        private bool ParseFormJson(String jsonData, JSON_QUEUE queue)
        {
            bool result = false;
            queue.SetAsChangeTrackingRoot();
            queue.SetAsUpdateOnSubmit();
            queue.PROCESS_START_DATE = DateTime.Now;
            try
            {
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                BprForm bprForm = serializer.Deserialize<BprForm>(jsonData);
                if (bprForm != null &&
                    !string.IsNullOrEmpty(bprForm._id))
                {
                    //The form                    
                    STD_FORM form = _formManager.GetFormByBprId(bprForm._id);
                    if (form == null)
                    {
                        form = new STD_FORM();
                        form.SetAsChangeTrackingRoot();
                        form.BPR_FORM_ID = bprForm._id;
                        form.CREATEDBY = "REST Service";
                        form.CREATED = DateTime.Now;
                        form.SetAsInsertOnSubmit();
                    }
                    else
                    {
                        form.SetAsChangeTrackingRoot();
                        form.UPDATED = DateTime.Now;
                        form.UPDATEDBY = "REST Service";
                        form.SetAsUpdateOnSubmit();
                        foreach (STD_FORM_SECTION formSection in form.STD_FORM_SECTIONs)
                        {
                            formSection.SetAsDeleteOnSubmit();
                        }
                        foreach (STD_FORM_SUB_SECTION subSection in form.STD_FORM_SUB_SECTIONs)
                        {
                            subSection.SetAsDeleteOnSubmit();
                        }
                        foreach (STD_FORM_QUESTION question in form.STD_FORM_QUESTIONs)
                        {
                            question.SetAsDeleteOnSubmit();
                        }
                        foreach (STD_FORM_ANSWER answer in form.STD_FORM_ANSWERs)
                        {
                            answer.SetAsDeleteOnSubmit();
                            foreach (STD_FORM_ANSWER_FIELD field in answer.STD_FORM_ANSWER_FIELDs)
                            {
                                field.SetAsDeleteOnSubmit();
                            }
                        }
                    }
                    form.NAME = bprForm.name;
                    form.OMB = bprForm.omb;
                    form.TITLE = bprForm.title;
                    form.VERSION = bprForm.version;
                    form.UPDATEDBY = "REST Service";
                    form.UPDATED = DateTime.Now;

                    //Form sections:
                    foreach (BprFormSection bprSection in bprForm.sections)
                    {
                        STD_FORM_SECTION formSection = new STD_FORM_SECTION();
                        formSection.BRP_FORM_SECTION_ID = String.IsNullOrEmpty(bprSection.sectionId) ? (int?)null : int.Parse(bprSection.sectionId);
                        formSection.LINE_NUMBER = String.IsNullOrEmpty(bprSection.lineNumber) ? (int?)null : int.Parse(bprSection.lineNumber);
                        formSection.CONCLUSION = String.IsNullOrEmpty(bprSection.conclusion) ? null : bprSection.conclusion;
                        formSection.DESCRIPTION = String.IsNullOrEmpty(bprSection.description) ? null : bprSection.description;
                        formSection.INTRODUCTION = String.IsNullOrEmpty(bprSection.introduction) ? null : bprSection.introduction;
                        formSection.NOTES = String.IsNullOrEmpty(bprSection.notes) ? null : bprSection.notes;
                        formSection.TITLE = String.IsNullOrEmpty(bprSection.title) ? null : bprSection.title;
                        formSection.CREATED = DateTime.Now;
                        formSection.UPDATED = DateTime.Now;
                        formSection.CREATEDBY = "REST Service";
                        formSection.UPDATEDBY = "REST Service";
                        form.STD_FORM_SECTIONs.Add(formSection);
                        formSection.SetAsInsertOnSubmit();

                        //Subsections:
                        foreach (BprFormSubsection bprSubSection in bprSection.subSections)
                        {
                            STD_FORM_SUB_SECTION subSection = new STD_FORM_SUB_SECTION();
                            subSection.BRP_FORM_SUB_SECTION_ID = String.IsNullOrEmpty(bprSubSection.subSectionId) ? (int?)null : int.Parse(bprSubSection.subSectionId);
                            subSection.TITLE = String.IsNullOrEmpty(bprSubSection.title) ? null : bprSubSection.title;
                            subSection.INTRODUCTION = String.IsNullOrEmpty(bprSubSection.introduction) ? null : bprSubSection.introduction;
                            subSection.CONCLUSION = String.IsNullOrEmpty(bprSubSection.conclusion) ? null : bprSubSection.conclusion;
                            subSection.TITLE = String.IsNullOrEmpty(bprSubSection.title) ? null : bprSubSection.title;
                            subSection.CREATED = DateTime.Now;
                            subSection.UPDATED = DateTime.Now;
                            subSection.CREATEDBY = "REST Service";
                            subSection.UPDATEDBY = "REST Service";
                            form.STD_FORM_SUB_SECTIONs.Add(subSection);
                            formSection.STD_FORM_SUB_SECTIONs.Add(subSection);
                            subSection.SetAsInsertOnSubmit();

                            //Questions:
                            foreach (BprFormQuestion bprQuestion in bprSubSection.questions)
                            {
                                STD_FORM_QUESTION question = new STD_FORM_QUESTION();
                                int parentId = int.MinValue;
                                int.TryParse(bprQuestion.parent, out parentId);
                                question.MAX_CHOICES = bprQuestion.maxChoices;
                                question.PARENT_ID = parentId == int.MinValue ? (int?)null : parentId;
                                question.BRP_FORM_QUESTION_ID = String.IsNullOrEmpty(bprQuestion.questionId) ? null : bprQuestion.questionId;
                                question.SOURCE_ID = String.IsNullOrEmpty(bprQuestion.sourceId) ? null : bprQuestion.sourceId;
                                question.TEXT = String.IsNullOrEmpty(bprQuestion.text) ? null : bprQuestion.text;
                                question.TYPE = String.IsNullOrEmpty(bprQuestion.type) ? null : bprQuestion.type;
                                question.MIN = String.IsNullOrEmpty(bprQuestion.min) ? "" : bprQuestion.min;
                                question.MAX = String.IsNullOrEmpty(bprQuestion.max) ? "" : bprQuestion.max;
                                question.PRE_REQ_OPERATOR = String.IsNullOrEmpty(bprQuestion.preReqOperator) ? null : bprQuestion.preReqOperator;
                                question.CREATED = DateTime.Now;
                                question.UPDATED = DateTime.Now;
                                question.CREATEDBY = "REST Service";
                                question.UPDATEDBY = "REST Service";
                                form.STD_FORM_QUESTIONs.Add(question);
                                formSection.STD_FORM_QUESTIONs.Add(question);
                                subSection.STD_FORM_QUESTIONs.Add(question);
                                question.SetAsInsertOnSubmit();

                                //Prerequisites:
                                /*foreach (BprFormPreReq bprPreReq in bprQuestion.preReq)
                                {
                                    STD_FORM_PRE_REQ_XREF preReq = new STD_FORM_PRE_REQ_XREF();
                                    preReq.STD_FORM_PRE_REQ_XREF_ID = String.IsNullOrEmpty(bprPreReq.questionId) ? 0 : int.Parse(bprPreReq.questionId);
                                    preReq.STD_FORM_ANSWER_ID_PRE_REQ = String.IsNullOrEmpty(bprPreReq.questionId) ? 0 : int.Parse(bprPreReq.questionId);
                                    preReq.CREATED = DateTime.Now;
                                    preReq.UPDATED = DateTime.Now;
                                    preReq.CREATEDBY = "REST Service";
                                    preReq.UPDATEDBY = "REST Service";
                                    preReq.SetAsInsertOnSubmit();
                                    question.STD_FORM_PRE_REQ_XREFs.Add(preReq);
                                }*/

                                //Answers:
                                foreach (BprFormAnswer bprAnswer in bprQuestion.answers)
                                {
                                    STD_FORM_ANSWER answer = new STD_FORM_ANSWER();
                                    answer.ORDER = bprAnswer.order;
                                    answer.EDITABLE = bprAnswer.editable;
                                    answer.BRP_FORM_ANSWER_ID = String.IsNullOrEmpty(bprAnswer.answerId) ? null : bprAnswer.answerId;
                                    answer.TEXT = String.IsNullOrEmpty(bprAnswer.text) ? null : bprAnswer.text;
                                    answer.HELP = String.IsNullOrEmpty(bprAnswer.help) ? "" : bprAnswer.help;
                                    answer.CREATED = DateTime.Now;
                                    answer.UPDATED = DateTime.Now;
                                    answer.CREATEDBY = "REST Service";
                                    answer.UPDATEDBY = "REST Service";
                                    form.STD_FORM_ANSWERs.Add(answer);
                                    formSection.STD_FORM_ANSWERs.Add(answer);
                                    subSection.STD_FORM_ANSWERs.Add(answer);
                                    question.STD_FORM_ANSWERs.Add(answer);
                                    answer.SetAsInsertOnSubmit();

                                    //Post Reqs:
                                    /*foreach (BprFormPostReq bprPostReq in bprAnswer.postReq)
                                    {
                                        STD_FORM_POST_REQ_XREF postReq = new STD_FORM_POST_REQ_XREF();
                                        postReq.STD_FORM_QUESTION_ID_POST_REQ = String.IsNullOrEmpty(bprPostReq.questionId) ? (int?)null : int.Parse(bprPostReq.questionId);
                                        postReq.CREATED = DateTime.Now;
                                        postReq.UPDATED = DateTime.Now;
                                        postReq.CREATEDBY = "REST Service";
                                        postReq.UPDATEDBY = "REST Service";
                                        postReq.SetAsInsertOnSubmit();
                                        answer.STD_FORM_POST_REQ_XREFs.Add(postReq);
                                    }*/

                                    //Fields:
                                    foreach (BprFormField bprField in bprAnswer.fields)
                                    {
                                        STD_FORM_ANSWER_FIELD field = new STD_FORM_ANSWER_FIELD();
                                        field.LABEL = String.IsNullOrEmpty(bprField.label) ? null : bprField.label;
                                        field.MAX = String.IsNullOrEmpty(bprField.max) ? null : bprField.max;
                                        field.MIN = String.IsNullOrEmpty(bprField.min) ? null : bprField.min;
                                        field.CREATED = DateTime.Now;
                                        field.UPDATED = DateTime.Now;
                                        field.CREATEDBY = "REST Service";
                                        field.UPDATEDBY = "REST Service";
                                        answer.STD_FORM_ANSWER_FIELDs.Add(field);
                                        field.SetAsInsertOnSubmit();
                                    }
                                }
                            }
                        }
                    }

                    // Save Form Info
                    using (_dbAhobpr = GetLocalContext(_connectionInfo))
                    {
                        _formManager.SetLoadWithForForms(_dbAhobpr);
                        result = (_formManager.UpdateForm(form) >= 0);


                        if (result)
                        {
                            queue.PROCESS_COMPLETE_DATE = DateTime.Now;
                            queue.PROCESS_STATUS = "success";
                            SaveJsonQueueData(queue);
                        }
                        else
                        {
                            MarkQueueEntryAsErrored(queue, "Failed to save form.");
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Parse JSON Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                result = false;
            }

            return result;
        }

        /// <summary>
        /// Get Json data by ID
        /// </summary>
        /// <param name="jsonId"></param>
        /// <returns></returns>
        private REGISTRANT_JSON GetJsonDataById(int jsonId)
        {
            REGISTRANT_JSON objReturn = null;

            try
            {
                using (AHOBPRLinqDataContext db = GetLocalContext(_connectionInfo))
                {
                    //call stored procedure
                    IList<REGISTRANT_JSON> results = db.SP_GetRegistrantJsonRow(jsonId).ToList();
                    if (results != null)
                    {
                        if (results.Count > 0)
                        {
                            objReturn = results.FirstOrDefault();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
            }

            return objReturn;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="jsonFollowupId"></param>
        /// <returns></returns>
        private REGISTRANT_JSON_FOLLOWUP GetJsonFollowupDataById(int jsonFollowupId)
        {
            REGISTRANT_JSON_FOLLOWUP objReturn = null;

            try
            {
                using (AHOBPRLinqDataContext db = GetLocalContext(_connectionInfo))
                {
                    //call stored procedure
                    IList<REGISTRANT_JSON_FOLLOWUP> results = db.SP_GetRegistrantJsonFollowupRow(jsonFollowupId).ToList();
                    if (results != null)
                    {
                        if (results.Count > 0)
                        {
                            objReturn = results.FirstOrDefault();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
            }

            return objReturn;
        }

        private BprInfo GetInfoJsonByResponseId(int responseRegistrantJsonId)
        {
            BprInfo info = null;
            try
            {
                REGISTRANT_JSON responseJsonObject = GetJsonDataById(responseRegistrantJsonId);
                if (responseJsonObject != null)
                {
                    REGISTRANT_JSON infoJsonObject = GetJsonData(responseJsonObject.OBJECT_ID, "info");
                    if (infoJsonObject != null)
                    {
                        string infoJsonData = infoJsonObject.JSON_DATA;
                        JavaScriptSerializer serializer = new JavaScriptSerializer();
                        info = serializer.Deserialize<BprInfo>(infoJsonData);
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
                return null;
            }

            return info;
        }


        /// <summary>
        /// Get REGISTRANT_JSON_ID from JSON_QUEUE_ID
        /// </summary>
        /// <param name="jsonQueueId"></param>
        /// <returns></returns>
        private JSON_QUEUE GetJsonQueue(int jsonQueueId)
        {
            JSON_QUEUE objReturn = null;

            try
            {
                using (AHOBPRLinqDataContext db = GetLocalContext(_connectionInfo))
                {
                    //call stored procedure
                    IList<JSON_QUEUE> results = db.SP_GetJsonQueueRow(jsonQueueId).ToList();
                    if (results != null)
                    {
                        if (results.Count > 0)
                        {
                            objReturn = results.FirstOrDefault();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
            }

            return objReturn;
        }

        /// <summary>
        /// Get REGISTRANT_JSON_ID from JSON_QUEUE_ID
        /// </summary>
        /// <param name="jsonQueueFollowupId"></param>
        /// <returns></returns>
        private JSON_QUEUE_FOLLOWUP GetJsonQueueFollowup(int jsonQueueFollowupId)
        {
            JSON_QUEUE_FOLLOWUP objReturn = null;

            try
            {
                using (AHOBPRLinqDataContext db = GetLocalContext(_connectionInfo))
                {
                    //call stored procedure
                    IList<JSON_QUEUE_FOLLOWUP> results = db.SP_GetJsonQueueFollowupRow(jsonQueueFollowupId).ToList();
                    if (results != null)
                    {
                        if (results.Count > 0)
                        {
                            objReturn = results.FirstOrDefault();
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
            }

            return objReturn;
        }

        #endregion

        #region Get JSON Follow-up data
        /// <summary>
        /// Get JSON data
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public List<string> GetJsonFollowups(string id, string type)
        {
            List<string> jsonData = new List<string>();
            if (string.IsNullOrEmpty(id) && string.IsNullOrEmpty(type))
            {
                return jsonData;
            }
            else
            {

                try
                {
                    Dictionary<String, Object> param = new Dictionary<string, object>();
                    param.Add("@ObjectType", type);
                    param.Add("@ObjectId", id);
                    List<Object[]> rows = DbUtil.ExecuteSProc("AHOBPR.SP_GetJsonFollowups", param);
                    foreach (Object[] row in rows)
                    {
                        jsonData.Add(row[1].ToString());
                    }

                    return jsonData;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
                    return null;
                }
            }
        }

        /// <summary>
        /// Get System Wide Follow-up Form Json
        /// </summary>
        /// <returns></returns>
        public List<string> GetSystemWideFormJsons()
        {
            List<string> jsonData = new List<string>();
            try
            {
                Dictionary<String, Object> param = new Dictionary<string, object>();
                List<Object[]> rows = DbUtil.ExecuteSProc("AHOBPR.SP_GetSystemWideFormJson", param);
                foreach (Object[] row in rows)
                {
                    jsonData.Add(row[1].ToString());
                }

                return jsonData;
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
                return null;
            }
        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <param name="type"></param>
        /// <param name="instanceId"></param>
        /// <returns></returns>
        public string GetJsonFollowup(string id, string type, string instanceId)
        {
            string jsonData = string.Empty;
            if (string.IsNullOrEmpty(id) && string.IsNullOrEmpty(type))
            {
                return jsonData;
            }
            else
            {

                try
                {
                    Dictionary<String, Object> param = new Dictionary<string, object>();
                    param.Add("@ObjectType", type);
                    param.Add("@ObjectId", id);
                    param.Add("@InstanceId", instanceId);
                    List<Object[]> rows = DbUtil.ExecuteSProc("AHOBPR.SP_GetJsonFollowup", param);
                    if (rows.Count > 0)
                    {
                        jsonData = rows[0][1].ToString();
                    }

                    return jsonData;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
                    return null;
                }
            }
        }

        #endregion

        #region Reopen Questionnaire

        /// <summary>
        /// 
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public bool reopenQuestionnaire(string id)
        {
            bool results = false;

            try
            {
                REGISTRANT_JSON registrantJson = this.GetJsonData(id, AHOBPRGlobal.AhobprJsonDataTypeResponses);
                if (registrantJson != null)
                {
                    JavaScriptSerializer serializer = new JavaScriptSerializer();
                    BprResponses responses = serializer.Deserialize<BprResponses>(registrantJson.JSON_DATA);
                    if (!string.IsNullOrEmpty(responses.responseStatus.questionnaireCompletedDate))
                    {
                        responses.responseStatus.questionnaireCompletedDate = string.Empty;
                        string jsonData = serializer.Serialize(responses);
                        registrantJson = SaveJsonData(id, AHOBPRGlobal.AhobprJsonDataTypeResponses, jsonData, string.Empty);
                        results = (registrantJson != null);
                    }
                    else
                    {
                        results = true;
                    }
                }
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
                results = false;
            }

            return results;
        }


        #endregion

        #region Get JSON data

        /// <summary>
        /// Get JSON data
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public REGISTRANT_JSON GetJsonData(string id, string type)
        {
            REGISTRANT_JSON jsonData = null;
            if (string.IsNullOrEmpty(id) && string.IsNullOrEmpty(type))
            {
                return jsonData;
            }
            else
            {

                if (type == AHOBPRGlobal.AhobprJsonDataTypeForm && string.IsNullOrEmpty(id))
                {
                    id = AHOBPRGlobal.AhobprDefaultFormId;
                }
                try
                {
                    using (var db = GetLocalContext(_connectionInfo))
                    {
                        jsonData = AHOBPRShared.GetCurrentJsonFromDb(id, type, db);
                    }
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "\r\n" + ex.StackTrace);
                }
            }

            return jsonData;
        }

        #endregion

        #region Save JSON data
        /// <summary>
        /// 
        /// </summary>
        /// <param name="objectId"></param>
        /// <param name="type"></param>
        /// <param name="data"></param>
        /// <param name="createdBy"></param>
        /// <returns></returns>
        public REGISTRANT_JSON SaveJsonData(string objectId, string type, string data, string createdBy)
        {
            return SaveJsonData(objectId, type, data, createdBy, 1);
        }

        public REGISTRANT_JSON SaveJsonData(string objectId, string type, string data, string createdBy, int tryCount)
        {
            if (tryCount >= 20)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                    "Unable to save JSON Data after 20 tries: {ObjectID:" + objectId + "}, {type:" + type + "},{data:" + data + "}");

                return null;
            }
            if (tryCount > 1)
            {
                _sharedManager.LogErrorMessage("Retry", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                    "retry #" + tryCount.ToString() + ": {ObjectID:" + objectId + "}");
            }
            REGISTRANT_JSON result = null;
            if (string.IsNullOrEmpty(objectId) || string.IsNullOrEmpty(type) || string.IsNullOrEmpty(data))
            {
                return result;
            }
            try
            {
                string createdByString = string.IsNullOrEmpty(createdBy) ? "RESTful Service" : createdBy;
                Dictionary<String, Object> param = new Dictionary<string, object>();
                param.Add("@objectId", objectId);
                param.Add("@objectType", type);
                param.Add("@jsonData", data);
                param.Add("@createdBy", createdByString);
                int registrantJsonId = int.Parse(DbUtil.ExecuteSProc("AHOBPR.sp_InsertRegistrantJson", param)[0][0].ToString());

                result = new REGISTRANT_JSON();
                result.OBJECT_ID = objectId;
                result.OBJECT_TYPE = type;
                result.JSON_DATA = data;
                result.CREATED = DateTime.Now;
                result.CREATEDBY = createdByString;
                result.UPDATED = DateTime.Now;
                result.UPDATEDBY = createdByString;
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "; Object: {ObjectID:" + objectId + "}, {type:" + type + "}; " + ex.StackTrace);
                result = SaveJsonData(objectId, type, data, createdBy, tryCount + 1);
            }
            return result;
        }

        public REGISTRANT_JSON SaveFormJasonData(string objectId, string sequenceId, string type, string data)
        {
            REGISTRANT_JSON result = null;

            if (string.IsNullOrEmpty(objectId) || string.IsNullOrEmpty(type) || string.IsNullOrEmpty(data) || string.IsNullOrEmpty(sequenceId))
            {
                return result;
            }

            if (sequenceId == "1")
            {
                result = SaveJsonData(objectId, type, data, string.Empty);
            }
            else
            {
                REGISTRANT_JSON json = GetJsonData(objectId, type);
                if (json != null)
                {
                    json.SetAsChangeTrackingRoot();
                    json.SetAsUpdateOnSubmit();
                    json.JSON_DATA = json.JSON_DATA + data;
                    json.UPDATED = DateTime.Now;
                    json.UPDATEDBY = "REST Service";

                    using (_dbAhobpr = GetLocalContext(_connectionInfo))
                    {
                        _dbAhobpr.DeferredLoadingEnabled = false;
                        //this line traverses all entities, attaching all of them as appropriate to the data context.
                        json.SynchroniseWithDataContext(_dbAhobpr);

                        //Check if any actual changes will occur
                        ChangeSet changeSet = _dbAhobpr.GetChangeSet();

                        if (changeSet.Updates.Count > 0)
                        {
                            try
                            {
                                _dbAhobpr.SubmitChanges(ConflictMode.ContinueOnConflict);
                                result = json;
                            }
                            catch (System.Data.SqlClient.SqlException ex)
                            {
                                _sharedManager.LogErrorMessage("Sql Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                                result = null;
                            }
                            catch (ChangeConflictException)
                            {
                                _dbAhobpr.ChangeConflicts.ResolveAll(RefreshMode.KeepCurrentValues);

                                try
                                {
                                    _dbAhobpr.SubmitChanges(ConflictMode.FailOnFirstConflict);
                                }
                                catch (Exception ex)
                                {
                                    _sharedManager.LogErrorMessage("Database Update", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message);
                                    result = null;
                                }
                            }
                            catch
                            {
                                result = null;
                            }
                        }
                    }
                }
            }
            return result;
        }

        public void SaveJsonQueueData(JSON_QUEUE queue)
        {
            try
            {
                Dictionary<String, Object> param = new Dictionary<string, object>();
                param.Add("@jsonQueueId", queue.JSON_QUEUE_ID);
                param.Add("@registrantJsonId", queue.REGISTRANT_JSON_ID);
                param.Add("@processStartDate", queue.PROCESS_START_DATE);
                param.Add("@processCompleteDate", queue.PROCESS_COMPLETE_DATE);
                param.Add("@processStatus", queue.PROCESS_STATUS == null ? " " : queue.PROCESS_STATUS);
                param.Add("@processError", queue.PROCESS_ERROR == null ? " " : queue.PROCESS_ERROR);
                param.Add("@createdBy", "RESTful Service");
                DbUtil.ExecuteSProc("AHOBPR.sp_InsertJsonQueue", param);
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "; " + ex.StackTrace);
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="queueFollowupStatus"></param>
        public void SaveJsonQueueFollowupData(JSON_QUEUE_FOLLOWUP_STATUS queueFollowupStatus)
        {
            try
            {
                Dictionary<String, Object> param = new Dictionary<string, object>();
                param.Add("@jsonQueueFollowupId", queueFollowupStatus.JSON_QUEUE_FOLLOWUP_ID);
                param.Add("@registrantJsonFollowupId", queueFollowupStatus.REGISTRANT_JSON_FOLLOWUP_ID);
                param.Add("@processStartDate", queueFollowupStatus.PROCESS_START_DATE);
                param.Add("@processCompleteDate", queueFollowupStatus.PROCESS_COMPLETE_DATE);
                param.Add("@processStatus", queueFollowupStatus.PROCESS_STATUS == null ? " " : queueFollowupStatus.PROCESS_STATUS);
                param.Add("@processError", queueFollowupStatus.PROCESS_ERROR == null ? " " : queueFollowupStatus.PROCESS_ERROR);
                param.Add("@createdBy", "RESTful Service");
                DbUtil.ExecuteSProc("AHOBPR.SP_InsertJsonQueueFollowup", param);
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "; " + ex.StackTrace);
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="objectId"></param>
        /// <param name="type"></param>
        /// <param name="instanceId"></param>
        /// <param name="data"></param>
        /// <param name="status"></param>
        /// <returns></returns>
        public REGISTRANT_JSON_FOLLOWUP SaveJsonFollowupData(string objectId, string type, string instanceId, string data, string status, string isSystemWide, string createdBy)
        {
            return SaveJsonFollowupData(objectId, type, instanceId, data, status, isSystemWide, createdBy, 1);
        }

        private REGISTRANT_JSON_FOLLOWUP SaveJsonFollowupData(string objectId, string type, string instanceId, string data, string status, string isSystemWide, string createdBy, int tryCount)
        {
            if (tryCount >= 20)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                    "Unable to save JSON Data followup after 20 tries: {ObjectID:" + objectId + "}, {type:" + type + "},{data:" + data + "}");

                return null;
            }
            if (tryCount > 1)
            {
                _sharedManager.LogErrorMessage("Retry", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name,
                    "retry #" + tryCount.ToString() + ": {ObjectID:" + objectId + "}");
            }
            REGISTRANT_JSON_FOLLOWUP result = null;
            if (string.IsNullOrEmpty(objectId) || string.IsNullOrEmpty(type) || string.IsNullOrEmpty(data))
            {
                return result;
            }
            try
            {
                string createdByString = string.IsNullOrEmpty(createdBy) ? "RESTful Service" : createdBy;
                Dictionary<String, Object> param = new Dictionary<string, object>();
                param.Add("@objectId", objectId);
                param.Add("@objectType", type);
                param.Add("@instanceId", instanceId);
                param.Add("@jsonData", data);
                param.Add("@status", status);
                param.Add("@createdBy", createdByString);
                param.Add("@systemWide", isSystemWide.ToLower() == "yes" ? true : false);
                int registrantJsonId = int.Parse(DbUtil.ExecuteSProc("AHOBPR.sp_InsertRegistrantJsonFollowup", param)[0][0].ToString());

                result = new REGISTRANT_JSON_FOLLOWUP();
                result.OBJECT_ID = objectId;
                result.OBJECT_TYPE = type;
                result.INSTANCE_ID = instanceId;
                result.JSON_DATA = data;
                result.STATUS = status;
                result.CREATED = DateTime.Now;
                result.CREATEDBY = createdByString;
                result.UPDATED = DateTime.Now;
                result.UPDATEDBY = createdByString;
                result.SYSTEM_WIDE_FLAG = isSystemWide.ToLower() == "yes" ? true : false;
            }
            catch (Exception ex)
            {
                _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "; Object: {ObjectID:" + objectId + "}, {type:" + type + "}; " + ex.StackTrace);
                result = SaveJsonFollowupData(objectId, type, instanceId, data, status, isSystemWide, createdBy, tryCount + 1);
            }
            return result;
        }


        /// <summary>
        /// Create a new message json for a registrant
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="emailtemplateid"></param>
        /// <returns></returns>
        public bool CreateMessageJson(string registrantId, string emailtemplateid, string createdBy)
        {
            bool result = false;

            if (!string.IsNullOrEmpty(registrantId) && !string.IsNullOrEmpty(emailtemplateid))
            {
                AHOBPRMessageManager messageManager = new AHOBPRMessageManager();
                EMAIL_LOG emailLog = messageManager.GetEmailLog(Convert.ToInt32(registrantId), Convert.ToInt32(emailtemplateid));
                string userId = _registrantManager.GetUserIdByRegistrantId(Convert.ToInt32(registrantId));
                if (emailLog == null)
                {
                    string registrantName = _registrantManager.SelectRegistrantNameById(Convert.ToInt32(registrantId));
                    AHOBPREmailTemplateManager templateManager = new AHOBPREmailTemplateManager();
                    EMAIL_TEMPLATE template = templateManager.GetById(Convert.ToInt32(emailtemplateid));
                    if (template != null)
                    {
                        EmailLogManager.LogMail(Convert.ToInt32(registrantId), string.Empty, string.Empty, string.Empty,
                            HttpUtility.HtmlDecode(template.SUBJECT),
                            HttpUtility.HtmlDecode(template.SALUTATION + Environment.NewLine + Environment.NewLine + template.BODY + Environment.NewLine + Environment.NewLine + template.CONTACT_INFO).Replace("{REGISTRANTNAME}"
                            , registrantName),
                            string.Empty, Convert.ToInt32(emailtemplateid), true, createdBy);
                    }
                    emailLog = messageManager.GetEmailLog(Convert.ToInt32(registrantId), Convert.ToInt32(emailtemplateid));
                }
                BprMessage bprMessage = new BprMessage();
                bprMessage._id = emailLog.EMAIL_ID.ToString();
                bprMessage.subject = emailLog.SUBJECT;
                bprMessage.body = emailLog.BODY;
                bprMessage.status = AHOBPRGlobal.AhobprJsonStatusNew;
                bprMessage.sentDate = emailLog.EMAIL_SENT.ToString();
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                string jsonData = serializer.Serialize(bprMessage);
                REGISTRANT_JSON_FOLLOWUP registrantJsonFollowup = SaveJsonFollowupData(userId, AHOBPRGlobal.AhobprJsonTypeMessage, bprMessage._id, jsonData, AHOBPRGlobal.AhobprJsonStatusNew, "no", createdBy);
                if (registrantJsonFollowup != null)
                {
                    result = true;
                }
            }

            return result;
        }

        /// <summary>
        /// Create Followup Form Json
        /// </summary>
        /// <param name="toolFormId"></param>
        /// <returns></returns>
        public bool CreateFollowupFormJson(string toolFormId, string isSystemWide, string createdBy)
        {
            bool result = false;
            STD_FORM form = _formManager.GetFormByToolFormId(toolFormId);
            if (form != null)
            {
                BprForm bprForm = CreateBprFormObject(form);
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                string jsonData = serializer.Serialize(bprForm);
                REGISTRANT_JSON_FOLLOWUP registrantJsonFollowup = SaveJsonFollowupData(bprForm._id, AHOBPRGlobal.AhobprJsonTypeFormFollowup, bprForm._id, jsonData, string.Empty, isSystemWide, createdBy);
                if (registrantJsonFollowup != null)
                {
                    result = true;
                }
            }

            return result;
        }

        /// <summary>
        /// Update follow-up form json with system wide flag
        /// </summary>
        /// <param name="toolFormId"></param>
        /// <returns></returns>
        public bool UpdateFollowupFormJsonSystemWide(string toolFormId)
        {
            bool result = false;
            STD_FORM form = _formManager.GetFormByToolFormId(toolFormId);
            if (form != null)
            {
                try
                {
                    Dictionary<String, Object> param = new Dictionary<string, object>();
                    param.Add("@objectId", form.BPR_FORM_ID);
                    param.Add("@objectType", AHOBPRGlobal.AhobprJsonTypeFormFollowup);
                    param.Add("@instanceId", form.BPR_FORM_ID);
                    DbUtil.ExecuteSProc("AHOBPR.SP_UpdateFormJsonFollowupSystemWide", param);
                    result = true;
                }
                catch (Exception ex)
                {
                    _sharedManager.LogErrorMessage("Exception", this.GetType().Name + "." + MethodBase.GetCurrentMethod().Name, ex.Message + "; Object: {ObjectID:" + form.BPR_FORM_ID + "}, {type:" + AHOBPRGlobal.AhobprJsonTypeFormFollowup + "}; " + ex.StackTrace);
                    result = false;
                }

            }

            return result;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="registrantId"></param>
        /// <param name="formId"></param>
        /// <returns></returns>
        public bool CreateUserFormJson(string registrantId, string formId, string createdBy)
        {
            bool result = false;
            if (!string.IsNullOrEmpty(registrantId) && !string.IsNullOrEmpty(formId))
            {
                string objectId = _registrantManager.GetUserIdByRegistrantId(Convert.ToInt32(registrantId));
                STD_FORM form = _formManager.GetFormByStdFormId(formId);
                if (!string.IsNullOrEmpty(objectId) && form != null)
                {
                    BprUserFollowupForm userForm = new BprUserFollowupForm();
                    userForm._id = form.BPR_FORM_ID;
                    userForm.name = form.NAME;
                    userForm.title = form.TITLE;
                    userForm.sentDate = DateTime.Now.ToString();
                    userForm.status = AHOBPRGlobal.AhobprJsonStatusNew;

                    JavaScriptSerializer serializer = new JavaScriptSerializer();
                    string jsonData = serializer.Serialize(userForm);
                    REGISTRANT_JSON_FOLLOWUP registrantJsonFollowup = SaveJsonFollowupData(objectId, AHOBPRGlobal.AhobprJsonTypeUserForm, userForm._id, jsonData, AHOBPRGlobal.AhobprJsonStatusNew, "no", createdBy);
                    if (registrantJsonFollowup != null)
                    {
                        result = true;
                    }
                }
            }


            return result;
        }

        private BprForm CreateBprFormObject(STD_FORM form)
        {
            BprForm bprForm = new BprForm();

            bprForm.name = form.NAME;
            bprForm.title = form.TITLE;
            bprForm.omb = form.OMB;
            bprForm.version = form.VERSION;
            bprForm._id = form.BPR_FORM_ID;

            //section
            BprFormSection bprSection = new BprFormSection();
            List<BprFormSection> bprSections = new List<BprFormSection>();
            STD_FORM_SECTION sqlSection = form.STD_FORM_SECTIONs.First();
            bprSection.sectionId = sqlSection.BRP_FORM_SECTION_ID.ToString();
            bprSection.title = sqlSection.TITLE;
            bprSections.Add(bprSection);
            bprForm.sections = bprSections.ToArray();

            //subSection
            BprFormSubsection bprSubsection = new BprFormSubsection();
            List<BprFormSubsection> bprSubsections = new List<BprFormSubsection>();
            STD_FORM_SUB_SECTION sqlSubsection = form.STD_FORM_SECTIONs.First().STD_FORM_SUB_SECTIONs.First();
            bprSubsection.title = sqlSubsection.TITLE;
            bprSubsection.subSectionId = sqlSubsection.BRP_FORM_SUB_SECTION_ID.ToString();
            bprSubsections.Add(bprSubsection);
            bprSection.subSections = bprSubsections.ToArray();

            //questions
            List<BprFormQuestion> bprQuestions = new List<BprFormQuestion>();
            foreach (STD_FORM_QUESTION sqlQuestion in sqlSubsection.STD_FORM_QUESTIONs)
            {
                BprFormQuestion bprQuestion = new BprFormQuestion();

                bprQuestion.questionId = sqlQuestion.BRP_FORM_QUESTION_ID;
                bprQuestion.text = sqlQuestion.TEXT;
                bprQuestion.type = sqlQuestion.TYPE;
                bprQuestion.min = sqlQuestion.MIN;
                bprQuestion.max = sqlQuestion.MAX;
                bprQuestion.maxChoices = sqlQuestion.MAX_CHOICES.HasValue ? (int)sqlQuestion.MAX_CHOICES : 1;

                bprQuestions.Add(bprQuestion);

                //answers
                List<BprFormAnswer> bprFormAnswers = new List<BprFormAnswer>();
                foreach (STD_FORM_ANSWER sqlAnswer in sqlQuestion.STD_FORM_ANSWERs)
                {
                    BprFormAnswer bprFormAnswer = new BprFormAnswer();
                    bprFormAnswer.answerId = sqlAnswer.BRP_FORM_ANSWER_ID;
                    bprFormAnswer.text = sqlAnswer.TEXT;
                    bprFormAnswer.help = sqlAnswer.HELP;
                    bprFormAnswer.editable = sqlAnswer.EDITABLE;
                    bprFormAnswer.order = sqlAnswer.ORDER;

                    bprFormAnswers.Add(bprFormAnswer);
                }
                bprQuestion.answers = bprFormAnswers.ToArray();
            }
            bprSubsection.questions = bprQuestions.ToArray();

            return bprForm;
        }

        #endregion
    }
}
